#include <windows.h>
#include <winsock.h>

#include <process.h>
#include <stdio.h>

#include "wcserver.h"
#pragma hdrstop

#include <math.h>
#include "thread.h"
#include "tttoe.h"

#define MAKEDWORD(a,b) ((DWORD)((DWORD)(a)<<16)|((DWORD)(b)))

SOCKET ListenSocket = INVALID_SOCKET;
HANDLE ListenThread = INVALID_HANDLE_VALUE;

///////////////////////////////////////////////////////////////////////////////
#define sVersion "1.00"

CRITICAL_SECTION cs;

struct TChanges { 
  int changer,changec,value; 
};

struct GameRecord {
  int GameStatus;
  int P1_Assigned;
  int P2_Assigned;
  int P1_Move;
  int P2_Move;
  int P1_PermissionToMove;
  int P2_PermissionToMove;
  char P1_Message_In_Bound[36];
  char P1_Message_Out_Bound[36];
  char P2_Message_In_Bound[36];
  char P2_Message_Out_Bound[36];
};

struct GameRecord GameList[25];   

class TCycler: public TThread {
public:
  TCycler();
  virtual ~TCycler();
protected:
  virtual BOOL CanClose() { return TRUE; }
  virtual BOOL Cycle() { return FALSE; }
  virtual void Go();
};

TCycler::TCycler()
{
  Start();
}

TCycler::~TCycler()
{
}

void TCycler::Go()
{
  while (Cycle());
}

///////////////////////////////////////////////////////////////////////////////

class TSocketCycler: public TCycler {
public:
  TSocketCycler(SOCKET s);
  virtual ~TSocketCycler();
protected:
  virtual int Send(void *buf, int len);
  virtual BOOL HandleIncoming(void *buf, int size) = 0;
  virtual BOOL Cycle();
protected:
  SOCKET Socket;
};

TSocketCycler::TSocketCycler(SOCKET s)
 : TCycler(), Socket(s)
{
}

TSocketCycler::~TSocketCycler()
{
  if (Socket != INVALID_SOCKET) {
    closesocket(Socket);
    Socket = INVALID_SOCKET;
    printf("We are in ~TSocketCycler....");
    printf("\r\n");
  }
}

BOOL TSocketCycler::Cycle()
{
  if (Socket != INVALID_SOCKET) {
    timeval tv;
    fd_set readset;
    FD_ZERO(&readset);
    FD_SET(Socket, &readset);
    ZeroMemory(&tv, sizeof(tv));
    if (select(0, &readset, NULL, NULL, &tv) > 0) {
      char buf[8192];
      int read = recv(Socket, buf, sizeof(buf), 0);
      if (!HandleIncoming(buf, read) && CanClose()) {
        printf(" Returning FALSE from TSocketCycler::Cycle");
        printf("\r\n");
        return FALSE;
      }
    }
  }
  return TRUE;
}

int TSocketCycler::Send(void *buf, int len)
{
  return ::send(Socket, (const char *)buf, len, 0);
}

///////////////////////////////////////////////////////////////////////////////

class TPacketCycler: public TSocketCycler {
 public:
  TPacketCycler(SOCKET s);
  virtual ~TPacketCycler();
protected:
  DWORD GlobalTop;
  char GlobalBuffer[8192];
protected:
  virtual BOOL Cycle();
  virtual BOOL HandleIncoming(void *buf, int size);
  virtual BOOL ProcessStructure(const TBaseStruct *bs) = 0;
};

TPacketCycler::TPacketCycler(SOCKET s)
 : TSocketCycler(s)
{
  GlobalTop = 0;
}

TPacketCycler::~TPacketCycler()
{
  printf("We are in the destructor....");
  printf("\r\n");
}

BOOL TPacketCycler::Cycle()
{
  BOOL ret = TSocketCycler::Cycle();

  // Check to see if we have enough in the buffer to extract a structure.
  // If the amount of data we have in our global buffer meets or exceeds
  // the size of our base structure AND the size of the structure we are 
  // trying to read, then pass the structre to the ProcessStructre 
  // procedure. Once the structure has been processed, we need to extract 
  // the structure from our gloabl buffer, and slide down any data beyond
  // where the structre ended.
  // 
  // Let me illistrate though how this is going to work. Each box below 
  // illistrates our global memory buffer. The data in the buffer is 
  // displayed as either an asterisk (*) or a lowercase oh (o). The unused 
  // portion of the buffer is illistrated through a period. 
  //
  //   [--------------------]  The first time into this routine, our buffer
  //   [oooooooooo..........]  will be completely blank. The data we just 
  //   [....................]  will be added to our global buffer (in this 
  //   [....................]  case 10 bytes). However, by inspecting the 
  //   [--------------------]  buffer, we determine that the structure we 
  //                           are trying to extract is 12 bytes long. We 
  // do not yet have enough in our buffer to process this structure. 
  // Therefore, we exit without acting upon the data in our global buffer.
  //    
  //   [--------------------]  The second time this routine is called, we 
  //   [**********oooooooooo]  have read an additional 15 bytes of data.
  //   [ooooo...............]  This is added to our gloabl memory buffer.
  //   [....................]  We now have enough in our buffer to extract
  //   [--------------------]  and proceess the 12 byte structure. Once 
  //                           extracted, however, we do not want to re-
  // process the same structure again. Therefore, we need to remove the 
  // twelve bytes comprising the structure, and slide down the data beyond
  // where the structure ends. 
  //
  //   [--------------------]  In this final example, we see that our buffer
  //   [*************ooooooo]  has been reset to not include the original 
  //   [ooo.................]  structure. In addition, we have received an 
  //   [....................]  additional 10 bytes. If the next structure in
  //   [--------------------]  is complete within our global buffer, we can 
  //                           go ahead and extract and process it. Otherwise,
  //  we will need to wait for one or more additional iterations.
  if (GlobalTop >= sizeof(TBaseStruct)) {
    TBaseStruct *bs = (TBaseStruct*)GlobalBuffer;
    ret &= ProcessStructure(bs);
    GlobalTop -= bs->size;
    memmove(GlobalBuffer, &GlobalBuffer[(int)bs->size], (size_t)GlobalTop);
  }
  return ret;
}

BOOL TPacketCycler::HandleIncoming(void *buf, int size)
{
  if (buf == NULL || size == 0) {
   return FALSE;
  }

  // add the new data we just received to our global memory buffer.
  if ((GlobalTop + size) < sizeof(GlobalBuffer)) {
    memcpy(&GlobalBuffer[GlobalTop], buf, size);
    GlobalTop += size;
    return TRUE;
  }
  return FALSE;
}

///////////////////////////////////////////////////////////////////////////////

class TGameCycler: public TPacketCycler {
public:
  TGameCycler(SOCKET s);
protected:
  virtual BOOL Cycle();
  virtual BOOL ProcessStructure(const TBaseStruct *bs);
  void HandlePlayerActions(struct TPlayerMove *MyPlayerInfo);
  void SubmitPlayerData(struct TPlayerMove *pm);
  void UpdatePlayerRecord(struct TPlayerMove *pm);
private:  
  BOOL GameOverCheck(struct TPlayerMove *pm);
  void TranslatePlayerMove(int pnum, int mvpos);
  void SetUpTheGame(struct TPlayerMove *pm);
  void MakePlayerOneMove(struct TPlayerMove *pm);
  void MakePlayerTwoMove(struct TPlayerMove *pm);
private:
  int   Depth;
  DWORD Count;
  BOOL  Verbose;
  int NumberOfMovesMade;
  BOOL GameOver;
  char Board[3][3];
  BOOL Done;
  int gr;
protected:
  TPlayerMove pm;
};

TGameCycler::TGameCycler(SOCKET s)
 : TPacketCycler(s)
{
Count = 0;
memset(&Board, 'A', sizeof(Board));
NumberOfMovesMade = 0;
}

/* sends data to the client */
void TGameCycler::SubmitPlayerData(struct TPlayerMove *pm)
{
  /* send the "pm" structure with all it's data to
     the TTT client.  The is a very good method for sending
     mixed data types to the client.
   */
  char mybuf1[2];
  memset(&mybuf1, 0, sizeof(mybuf1)); 
  _itoa(pm->PlayerNumber, mybuf1, 10);
  printf("Sending data to Player Number ");
  printf(mybuf1);
  printf(" TTT client....");
  printf("\r\n");

  send(Socket, (const char *)pm, sizeof(TPlayerMove), 0);
  
}

void TGameCycler::UpdatePlayerRecord(struct TPlayerMove *pm)
{
  /* in this procedure we are updating the "pm" structure
     to reflect all current data contained in our global
     variables for this game.
   */

  pm->IsThereAWinnerYet = GameList[pm->GameNumber].GameStatus;
  if(pm->PlayerNumber == 1){
    pm->PermissionToMove = GameList[pm->GameNumber].P1_PermissionToMove;
    pm->MovePosition = GameList[pm->GameNumber].P1_Move;
    pm->OMovePosition = GameList[pm->GameNumber].P2_Move;
    strcpy(pm->TTTServiceMessage, GameList[pm->GameNumber].P1_Message_Out_Bound);
  }
  if(pm->PlayerNumber == 2){
    pm->PermissionToMove = GameList[pm->GameNumber].P2_PermissionToMove;
    pm->MovePosition = GameList[pm->GameNumber].P2_Move;
    pm->OMovePosition = GameList[pm->GameNumber].P1_Move;
    strcpy(pm->TTTServiceMessage, GameList[pm->GameNumber].P2_Message_Out_Bound);
  }
}

BOOL TGameCycler::Cycle()
{

  /* when data is not being received this function will
     cycle constantly to check the state of the global
     variables we are interested in watching.  If there
     is a change we will do whatever processing we need to
     do and then send a data structure to the appropriate
     player to let him know it's his turn or he can move or
     whatever.  This method ensures that our client does not
     have to poll constantly in order just to find out if
     something has changed.  Use this method to keep the
     service in control of moves, actions, etc..
   */
  BOOL ret = TPacketCycler::Cycle();

  if ((ret != FALSE) && (GameList[pm.GameNumber].GameStatus != 0)){
   /* checks for permission for Player 2 to move and if player
      has received it, then sends notification to him.
    */
   if((GameList[pm.GameNumber].P2_PermissionToMove == 1) &&
     (pm.PlayerNumber == 2)){
       UpdatePlayerRecord(&pm);
       char mybuf2[2];
       memset(&mybuf2, 0, sizeof(mybuf2)); 
       _itoa(pm.PlayerNumber, mybuf2, 10);
       printf("\r\n");
       printf("Sending turn notification to Player Number ");
       printf(mybuf2);
       printf("\r\n");
       SubmitPlayerData(&pm);
       /* OK he's been notified so change the flag
          to show a waiting state and make sure we
          don't repeat this again
        */
       GameList[pm.GameNumber].P2_PermissionToMove = 3;
   }
   /* checks for permission for Player 1 to move and if player
      has received it, then sends notification to him.
    */
   if((GameList[pm.GameNumber].P1_PermissionToMove == 1) &&
     (pm.PlayerNumber == 1)){
       UpdatePlayerRecord(&pm);
       char mybuf3[2];
       memset(&mybuf3, 0, sizeof(mybuf3)); 
       _itoa(pm.PlayerNumber, mybuf3, 10);
       printf("\r\n");
       printf("Sending turn notification to Player Number ");
       printf(mybuf3);
       printf("\r\n");
       SubmitPlayerData(&pm);
       /* OK he's been notified so change the flag
          to show a waiting state and make sure we
          don't repeat this again
        */
       GameList[pm.GameNumber].P1_PermissionToMove = 3;
   }
}
  /* if the game is over then break out of the cycle loop */
  if((GameList[pm.GameNumber].GameStatus == 3) && 
      (pm.PlayerNumber != 0)){
       char mybuf15[2];
       memset(&mybuf15, 0, sizeof(mybuf15));
       printf("Player Number ");
       _itoa(pm.PlayerNumber, mybuf15, 10);
       printf(mybuf15); 
       printf(" sending FALSE from game CYCLE....");
       printf("\r\n");
   ret = FALSE;
  }
  return ret;
}


void TGameCycler::HandlePlayerActions(struct TPlayerMove *MyPlayerInfo)
{
 // see the TTTOE.H unit for further info on the structure

 /* transfer data from our temporary structure called
    "MyPlayerInfo" to a permanent data structure called "pm"
  */
 pm.type = etPlayerMove;
 pm.size = sizeof(TPlayerMove); 
 pm.GameNumber = MyPlayerInfo->GameNumber;
 pm.PlayerNumber = MyPlayerInfo->PlayerNumber;
 pm.PermissionToMove = MyPlayerInfo->PermissionToMove;
 pm.IsThereAWinnerYet = MyPlayerInfo->IsThereAWinnerYet;
 pm.MovePosition = MyPlayerInfo->MovePosition;
 pm.OMovePosition = MyPlayerInfo->OMovePosition;
 strcpy(pm.TTTServiceMessage, MyPlayerInfo->TTTServiceMessage);
 
 char mybuf4[2];
 memset(&mybuf4, 0, sizeof(mybuf4)); 
 _itoa(pm.PlayerNumber, mybuf4, 10);
 printf("Receiving data from Player Number ");
 printf(mybuf4);
 printf(" TTT client....");
 printf("\r\n");
  
 /* if player 2 can move */
 if((GameList[pm.GameNumber].P2_PermissionToMove == 3) &&
     (pm.PlayerNumber == 2)){
    MakePlayerTwoMove(&pm);
 }

 /* if player 1 can move */
 if((GameList[pm.GameNumber].P1_PermissionToMove == 3) &&
     (pm.PlayerNumber == 1)){
   MakePlayerOneMove(&pm);
 } 
 /* if new, then set up the game for each incoming player */  
 if(pm.GameNumber == 99){
   SetUpTheGame(&pm);
 }
}

void TGameCycler::SetUpTheGame(struct TPlayerMove *pm)
{
  int q = 0;
  int q1 = 0;
  BOOL SpotFound = FALSE;
  int WhichPlayer = 0;
  
  /* checks the game structure array for a player slot */  
  while(!SpotFound){
  
    /* first checks for a game that has no 2nd player yet */
    if((GameList[q].GameStatus == 1) && (GameList[q].P2_Assigned == 0)
      && (!SpotFound)){
      WhichPlayer = 2;
      gr = q;
      SpotFound = TRUE;  
    }

    /* starts a new game in an unused slot */   
    if((GameList[q].GameStatus == 0) && (!SpotFound)){
      WhichPlayer = 1;
      gr = q;
      SpotFound = TRUE;
    }

    /* checks for a game slot recently vacated */
    if((GameList[q].GameStatus == 3) && (!SpotFound)){
      WhichPlayer = 1;
      gr = q;
      SpotFound = TRUE;
   
      EnterCriticalSection(&cs);

      GameList[q].GameStatus = 0;
      GameList[q].P1_Assigned = 0;
      GameList[q].P2_Assigned = 0;
      GameList[q].P1_Move = 0;
      GameList[q].P2_Move = 0;
      GameList[q].P1_PermissionToMove = 0;
      GameList[q].P2_PermissionToMove = 0;
      memset(GameList[q].P1_Message_In_Bound, 0, 
       sizeof(GameList[q].P1_Message_In_Bound));   
      memset(GameList[q].P2_Message_In_Bound, 0, 
       sizeof(GameList[q].P2_Message_In_Bound));
      memset(GameList[q].P1_Message_Out_Bound, 0, 
       sizeof(GameList[q].P1_Message_Out_Bound));
      memset(GameList[q].P2_Message_Out_Bound, 0, 
       sizeof(GameList[q].P2_Message_Out_Bound));

      LeaveCriticalSection(&cs);
      char mybuf5[3];
      memset(&mybuf5, 0, sizeof(mybuf5));
      _itoa(q, mybuf5, 10);
      printf("Reclaimed slot number ");
      printf(mybuf5);
      printf("\r\n");    
    }

    if(!SpotFound){q++;} 
    }

 pm->GameNumber = gr;
 pm->PlayerNumber = WhichPlayer;

  /* The thread entering after the EnterCriticalSection
     call has exclusive rights to all code and data contained
     therein until a LeaveCriticalSection call is encountered.
     This is similar in concept to DOS record/file locking.
     Any other threads attempting to enter here while another
     thread is present will be temporarily suspended until the
     owning thread leaves and makes the LeaveCriticalSection call.
     At that point the suspended thread will be activated and
     allowed to enter.  Use this to control access to global
     data shared by threads.
   */
 EnterCriticalSection(&cs);

 if(WhichPlayer == 1){
  /* indicate in global data slot that player 1 is assigned */
  GameList[gr].P1_Assigned = 1;
  /* copy message to global data slot */
  strcpy(GameList[gr].P1_Message_Out_Bound, "Welcome! Looking for Player 2!");
  /* transfer permission status information to the global data slot */
  GameList[gr].P1_PermissionToMove = 2;
  /* game waiting flag global data slot updated */
  GameList[gr].GameStatus = 1;
  /* update global data slot on player 1 info */
  GameList[gr].P1_Move = 0;
  UpdatePlayerRecord(pm);
  char mybuf6[3];
  char mybuf7[2];
  memset(&mybuf6, 0, sizeof(mybuf6));
  memset(&mybuf7, 0, sizeof(mybuf7));
  printf("\r\n");
  _itoa(pm->GameNumber, mybuf6, 10);
  printf("Game Number ");
  printf(mybuf6);
  _itoa(pm->PlayerNumber, mybuf7, 10);
  printf(" set up for Player Number ");
  printf(mybuf7);
  printf("\r\n");  
  SubmitPlayerData(pm);
 }

 if(WhichPlayer == 2){
   /* indicate in global data slot that player 2 is assigned */
  GameList[gr].P2_Assigned = 1;
  /* copy messages to global data slots */
  strcpy(GameList[gr].P1_Message_Out_Bound, "Game started - Opponents move!");
  strcpy(GameList[gr].P2_Message_Out_Bound, "Game started - Your move!");
  /* transfer who has permission to move to global data slot */
  GameList[gr].P2_PermissionToMove = 1;
  /* game in progress flag global data slot updated */
  GameList[gr].GameStatus = 2;
  /* update global data slot on player 2 info */
  GameList[gr].P2_Move = 0;
  UpdatePlayerRecord(pm);
  char mybuf8[3];
  char mybuf9[2];
  memset(&mybuf8, 0, sizeof(mybuf8));
  memset(&mybuf9, 0, sizeof(mybuf9));
  printf("\r\n");
  _itoa(pm->GameNumber, mybuf8, 10);
  printf("Game Number ");
  printf(mybuf8);
  _itoa(pm->PlayerNumber, mybuf9, 10);
  printf(" set up for Player Number ");
  printf(mybuf9);
  printf("\r\n");
  SubmitPlayerData(pm);
  /* we have sent notification it's his turn - now we wait */
  GameList[gr].P2_PermissionToMove = 3;
 }
 
 /* we are done now so other threads may now enter and do work */
 LeaveCriticalSection(&cs);
 
}

void TGameCycler::MakePlayerOneMove(struct TPlayerMove *pm)
{

    /* Handles player one actions */

    printf("Entering MakePlayerOneMove function..");
    printf("\r\n");
    
    EnterCriticalSection(&cs);
     
    GameList[pm->GameNumber].P1_Move = pm->MovePosition;
    TranslatePlayerMove(pm->PlayerNumber,
      GameList[pm->GameNumber].P1_Move);

    TranslatePlayerMove(2,
      GameList[pm->GameNumber].P2_Move);
 
    strcpy(GameList[gr].P1_Message_In_Bound, pm->TTTServiceMessage);
    strcpy(GameList[gr].P2_Message_Out_Bound, GameList[gr].P1_Message_In_Bound);
    GameOver = FALSE; 
    GameOver = GameOverCheck(pm);
    GameList[pm->GameNumber].P1_PermissionToMove = 2;
    GameList[pm->GameNumber].P2_PermissionToMove = 1;

    LeaveCriticalSection(&cs);
    
    if(GameOver){
     UpdatePlayerRecord(pm);
     SubmitPlayerData(pm);
    }
    printf("Leaving MakePlayerOneMove function..");
    printf("\r\n");
}

void TGameCycler::MakePlayerTwoMove(struct TPlayerMove *pm)
{
    /* Handles player two actions */
    NumberOfMovesMade++;

    printf("Entering MakePlayerTwoMove function..");
    printf("\r\n");
  
    EnterCriticalSection(&cs);

    GameList[pm->GameNumber].P2_Move = pm->MovePosition;
    TranslatePlayerMove(1,
      GameList[pm->GameNumber].P1_Move);

    TranslatePlayerMove(pm->PlayerNumber,
      GameList[pm->GameNumber].P2_Move);    

    strcpy(GameList[gr].P2_Message_In_Bound, pm->TTTServiceMessage);
    strcpy(GameList[gr].P1_Message_Out_Bound, GameList[gr].P2_Message_In_Bound);
    GameOver = FALSE; 
    GameOver = GameOverCheck(pm);
    GameList[pm->GameNumber].P2_PermissionToMove = 2;
    GameList[pm->GameNumber].P1_PermissionToMove = 1;
    LeaveCriticalSection(&cs);
    if(GameOver){
      UpdatePlayerRecord(pm);
      SubmitPlayerData(pm);
    }
    printf("Leaving MakePlayerTwoMove function..");
    printf("\r\n");
}

/* translates move from player to the game board array
   location.
 */
void TGameCycler::TranslatePlayerMove(int pnum, int mvpos)
{
	char pn = 'A';
  if(pnum == 1){ pn = 'X'; }
  if(pnum == 2){ pn = 'O'; }
  switch(mvpos){
	 case 0:{
     return;
   }
   case 1:{
     Board[0][0] = pn;
     return;
	 }
	 case 2:{
     Board[0][1] = pn;
     return;
	 }
	 case 3:{
     Board[0][2] = pn;
     return;
	 }
	 case 4:{
     Board[1][0] = pn;
     return;
	 }
	 case 5:{
     Board[1][1] = pn;
     return;
	 }
	 case 6:{
     Board[1][2] = pn;
     return;
	 }
	 case 7:{
     Board[2][0] = pn;
     return;
	 }
	 case 8:{
     Board[2][1] = pn;
     return;
	 }
	 case 9:{
     Board[2][2] = pn;
     return;
	 }
 }
}

BOOL TGameCycler::ProcessStructure(const TBaseStruct *bs)
{
  if (bs == NULL) {
    return FALSE;
  }
  switch (bs->type) {
    case etPlayerMove: {// this type structure arrived so process it
      if (bs->size == sizeof(TPlayerMove)) {
        TPlayerMove *MyPlayerInfo = (TPlayerMove*)bs;
         HandlePlayerActions(MyPlayerInfo);  
      }
      break;
    }
    case etPlayerMessage: {// this type structure arrived so process it
      if (bs->size == sizeof(TPlayerMove)) {
      }
      break;
    }
  }
 return TRUE;
}

BOOL TGameCycler::GameOverCheck(struct TPlayerMove *pm)
{ 
  /* The code in this section comes from the Book
     Black Art of Windows Game Programming - Developing
     High Speed Games with WinG by Eric R. Lyons
     and published  by The Waite Group - ISBN 1-878739-95-6
     Copyright 1995.  A very good source of general
     windows game programming information!
   */

  int g,h,testor,testoc,testxr,testxc, score = 0;
  BOOL ItsOver = FALSE;


  //check rows and columns simultaneously
  for (g=0;g<3;++g) { 
    testxr = testxc = testor = testoc = 0;
	  for (h=0;h<3;++h) { 
      if (Board[g][h] == 'X') testxr++; 
      if (Board[g][h] == 'O') testor++;
      if (Board[h][g] == 'X') testxc++;
      if (Board[h][g] == 'O') testoc++;
    }

		// add to score according to heuristic
    if (testxr && !testor) score++;
    if (testor && !testxr) score--;
    if (testxc && !testoc) score++;
    if (testoc && !testxc) score--;   

		// any wins?
    if (( testxr == 3) || (testxc == 3)) score += 98;
    if (( testor == 3) || (testoc == 3)) score -= 98;
  }

	// nasty check of diagonals
  testor = (Board[1][1] == 'O') +
           (Board[0][0] == 'O') +
           (Board[2][2] == 'O');
  testxr = (Board[1][1] == 'X') +
           (Board[0][0] == 'X') +
           (Board[2][2] == 'X');
  
  testoc = (Board[1][1] == 'O') +
           (Board[2][0] == 'O') +
           (Board[0][2] == 'O');
  testxc = (Board[1][1] == 'X') +
           (Board[2][0] == 'X') +
           (Board[0][2] == 'X');

	// add to score
  /* if there are x's in the row but no o's then increment*/
  if (testxr && !testor) score++;
  /* if there are o's in the row but no x's then decrement*/
  if (testor && !testxr) score--;
  /* if there are x's in the column but no o's then increment*/
  if (testxc && !testoc) score++;
  /* if there are o's in the column but no x's then decrement*/
  if (testoc && !testxc) score--;
  
  if ((testxr == 3) || (testxc == 3)) score += 98;
 
  
  if ((testor == 3) || (testoc == 3)) score -= 98;
    
  if((NumberOfMovesMade > 4) || (score > 50) || (score < -50)){
     GameList[pm->GameNumber].GameStatus = 3;  // game is over
  }

  char mybuf10[4];
  memset(&mybuf10, 0, sizeof(mybuf10)); 
  _itoa(score, mybuf10, 10);
  printf("The score is: ");
  printf(mybuf10);
  printf("\r\n");

  if(GameList[pm->GameNumber].GameStatus == 3){
    ItsOver = TRUE;
    printf("Game over!!");
    printf("\r\n");
  }    
     
  return ItsOver;
}

int main(int, char *[])
{
  // Connect to the default wildcat server

  if (!WildcatServerConnect(NULL)) {
    return 1;
  }
  if (!WildcatServerCreateContext()) {
    return 1;
  }

  // Initialize and load WinSock - establish service socket

  WSAData wsaData;
  WSAStartup(MAKEWORD(1, 1), &wsaData);
  SOCKET s = socket(AF_INET, SOCK_STREAM, 0);
  sockaddr_in sin;
  sin.sin_family = PF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = 0;
  bind(s, (sockaddr *)&sin, sizeof(sin));
  int n = sizeof(sin);
  getsockname(s, (sockaddr *)&sin, &n);

  // Register service with Wildcat!

  TWildcatService service;
  ZeroMemory(&service, sizeof(service));
  strcpy(service.Name, "Tic Tac Toe");
  strcpy(service.Vendor, "Mustang Software, Inc.");
  service.Version = 1;
  service.Port = ntohs(sin.sin_port);
  RegisterService(service);

  printf("Tic Tac Toe service active on %s/%d\n", inet_ntoa(*(struct in_addr *)&service.Address), ntohs(sin.sin_port));
  listen(s, 5);
 
  /* creates a critical section object named "cs" */
  InitializeCriticalSection(&cs);

  int q;
  for (q=0;q<25;++q) {
  /* 0 for not yet playing, 1 for waiting for another player
     2 for game in play, 3 game is over
   */ 
   GameList[q].GameStatus = 0;
   /* 0 for yes, 1 for no */
   GameList[q].P1_Assigned = 0;
   /* 0 for yes, 1 for no */
   GameList[q].P2_Assigned = 0;
   /*  0 for no move, 1 thru 9 for selected board spot */
   GameList[q].P1_Move = 0;
   /*  0 for no move, 1 thru 9 for selected board spot */
   GameList[q].P2_Move = 0;
   /* who can move now:  0 for none, 1 for Player 1,
      2 for player 2
    */ 
   GameList[q].P1_PermissionToMove = 0;
   GameList[q].P2_PermissionToMove = 0;
   /* messages to and from players */
   memset(GameList[q].P1_Message_In_Bound, 0, 
     sizeof(GameList[q].P1_Message_In_Bound));   
   memset(GameList[q].P2_Message_In_Bound, 0, 
     sizeof(GameList[q].P2_Message_In_Bound));
   memset(GameList[q].P1_Message_Out_Bound, 0, 
     sizeof(GameList[q].P1_Message_Out_Bound));
   memset(GameList[q].P2_Message_Out_Bound, 0, 
     sizeof(GameList[q].P2_Message_Out_Bound));
  }

  /* the below "while" statement is nothing but an
     endless loop which listens to a socket for incoming
     requests.
   */

  while (1) { 
    sockaddr_in tin;
    int n = sizeof(tin);
    SOCKET t = accept(s, (sockaddr *)&tin, &n);
    if (t != INVALID_SOCKET) {
      new TGameCycler(t);      
    }
  }
  
  /* deletes the "cs" object which is no longer needed */
  DeleteCriticalSection(&cs);
  return 0;
}