/****************************************************************************
 *      (c) 1991    Corporation of the President of the Church of Jesus Christ
 *      Copyright   of Latter-day Saints.
 *      All rights reserved.  Unauthorized reproduction of this software is
 *      prohibited and is in violaton of United States copyright laws.
 ****************************************************************************/

/*****************************************************************************
*
*   FILE NAME:          GEDPROC.C
*
*   DESCRIPTION:
*   The purpose of the GEDCOM processor is to visit each record and line in 
*   a GEDCOM file, matching tags in a context-sensitive fashion and calling 
*   corresponding user-/defined functions to process the information.
* 
*   MODIFICATION HISTORY:
*   -------------------------------------------------------------------------
*   |Date     | Who | Comments
*   -------------------------------------------------------------------------
*   |SEP    90| LAM | Program writen in C
*   -------------------------------------------------------------------------
*   |NOV 15 90| LAM | Optimizing, changed variable names, updated comments
*   -------------------------------------------------------------------------
*   |  "    " | LAM | Took out cmpWrd function. Now using ged_match
*   -------------------------------------------------------------------------
*   |DEC 3  90| LAM | ged_error now returns an integer.
*   -------------------------------------------------------------------------
*   |DEC 19 90| LAM | bad function names called from grammar are now flagged.
*   -------------------------------------------------------------------------
*   |MAR    90| LAM | Added the ged_traverse_grammar() function.
*   -------------------------------------------------------------------------
*   |09/04/91 10:56am| LAM | Recoded ged_error codes
*   -------------------------------------------------------------------------
*
*****************************************************************************/
#ifdef CYBER
#include ":nve.aimmgr.gedcom_c2.gedcom_h" /* gedcom_h must be defined 1st */
/* #include "gedproc_h" */
#else
#include "gedcom.h"
/* #include "gedproc.h" */
#include "gedutil.h"
#endif


struct funcReg { /* Used for registering user-defined functions */
   byte name[FUNC_NAME_SIZ + 1]; /* The grammar calls a func using a string
				  * which matches with this */
#ifdef CYBER
   int (*funcPointer)();  /* pointer to the function to call */
#else
int (*funcPointer)(NODE *, byte *);  /* pointer to the function to call */
#endif
   } funcsArray[MAX_FNCS]; /* This array is searched each time a user-defined
			    * function is called */

static int numFuncsUsed; /* The acual number of user-defined functions declared*/


/**************************************************************************
 *
 * NAME:  ged_process
 *
 * DESCRIPTION:  This function will load in the grammar and process the data
 *               based on the instructions in the grammar.
 *               Currently GLIMIT is defined as (unsigned long) and
 *               TLIMIT is (unsigned long). See gedproc.h for defines.  You
 *               may want to write your own ged_process function if this does
 *               not mmet your needs.
 *
 * CALLED BY:    user
 *
 **************************************************************************/
int ged_process(fg, ft)
    FILE *fg, /* Pointer to grammar file. */
	 *ft; /* Pointer to GEDCOM data tree. */
  {
  POOL *dataPool,    /* Pool for GEDCOM data records */
      *grammarPool; /* Pool for the whole grammar */
  NODE *grammar,     /* The whole grammar tree */
      *data;        /* Data tree.  Will hold one data record (between 0 levels).*/
  short status=MORE_RECORDS; /* Status of ged_read_tree function call */
  int errStat; /* error code returned by user-defined functions and ged_error */
  int endLevel = 0; /* endlevel used only when partial read flag set to 1 */
  byte *endTag = NULL;
  int partial = 0;   /*  read tree set to read normal 0 level records */

  grammarPool = ged_set_pool(ged_open_pool((unsigned)GLIMIT, sizeof(NODE)));
  grammar = ged_load_grammar(fg);

  /* Create a new pool for the GEDCOM data */
  dataPool = ged_set_pool(ged_open_pool((unsigned)TLIMIT, sizeof(NODE)));

  while ((status != LAST_RECORD) && ((data = read_tree(ft, &status, GLIMIT, endLevel, endTag, partial))
    !=(NODE *)NULL)&& (status != END_OF_FILE))
    {
    if ((errStat = ged_traverse(data, grammar)) >= 200) /* process the data */
        break; /* discontinue if user errStat >= 200 */
    GED_RESET_POOL(); 
    }
  GED_DESTROY_POOL(); /* Processing finished. Destroy pools and return */
  ged_set_pool(grammarPool); /* destroy the grammar pool */
  GED_DESTROY_POOL();

  /* if no fatal error, return status*/
  return((errStat >= 100) ? errStat: (int)status); /* return END_OF_FILE or errStat */
  }

/**************************************************************************
 *
 * NAME: ged_traverse_grammar
 *
 * DESCRIPTION: recursive function that traverses the grammar and data trees.
 *              dataCtxt and grammarCtxt are the current levels of the
 *              respective trees. This is similar to ged_traverse(). Difference is:
 *              this is driven by grammar, where ged_traverse() is driven by the data
 *              tree.  The effect is that with this function you have control
 *              over when a function (hook) is called, conversly, ged_traverse() will
 *              call a hook only when it happens to get around to parsing the hook's
 *              parent tag. NOTE: dataCtxt must have an extra node
 *              added onto the root of the tree; grammarCtxt does not need this.
 *              (Its the opposite with ged_traverse().  This function should not to be used
 *              where the dataCtxt tree has siblings with the same tag--it's
 *              ambiguous.
 * 
 *              This function is not documented in the GEDCOM library docs because
 *              currently, this function will only process the current record that
 *              is loaded in from a GEDCOM file.  A future version will hunt for
 *              a FAM record if the current node in grammarCtxt contains a FAM tag.
 *
 * 
 **************************************************************************/
int ged_traverse_grammar(dataCtxt, grammarCtxt)
    NODE *dataCtxt, /* Current node of data tree */
         *grammarCtxt; /* Current node of grammar tree */ 
    {
    NODE *currentDataCtxt; /* The tag in the data's sibling list that matches
                            * the grammarCtxt tag */
    int errStat= DFLT_ERR; /* Status code returned by user-defined functions &
                            * ged_error */ 

    /* See if tag is valid */
    do{
    /*Search for the tag in data's sibling list that matches the grammarCtxt tag*/
      if (currentDataCtxt = huntForTags(grammarCtxt,dataCtxt))
        {
        do
	  {
          /* perform preprocess semantic function */
          if (grammarCtxt->child &&
            ( errStat = preOrPostProcess(grammarCtxt->child, currentDataCtxt, '-') ) 
              >= 100) return(errStat);

          /* Process children */
          if (grammarCtxt->child &&
            ( errStat = ged_traverse_grammar(currentDataCtxt, grammarCtxt->child)) >= 100 )
              return(errStat);

	  /* perform postprocess semantic function  */
          if (grammarCtxt->child &&
            (errStat = preOrPostProcess(grammarCtxt->child, currentDataCtxt, '+')) >= 100)
            return(errStat);
          }while ( currentDataCtxt = huntForMoreTags(grammarCtxt, currentDataCtxt) );
	}
      else 
	  {
          /*Skips children. ged_error: Tag Not Found */
          if ((errStat = ged_error(302, dataCtxt)) >= 100)
            break;
          if (errStat ==errStat ) /* Need to define what this value is, unless always executed */
            if ((errStat = traverse_solo(grammarCtxt)) >= 100 )
              return(errStat);
          }
      } while (grammarCtxt = grammarCtxt->sibling);
    return(errStat);
    }

/**************************************************************************
 *
 * NAME: traverse_solo
 *
 * DESCRIPTION: Same as ged_traverse_grammar but this only ged_traverses grammarCtxt.
 * This is used when there is not a tag in dataCtxt that matches grammarCtxt.
 * The hooks are still called, send a NULL argument to the dataCtxt parameter.
 *
 * CALLED BY:   ged_traverse_grammar and recursively
 * 
 *ged_ged_*************************************************************************/
int traverse_solo(grammarCtxt)
    NODE *grammarCtxt; /* Current node of grammar tree */ 
  {
  int errStat= DFLT_ERR; /* Status code returned by user-defined functions &
                          * ged_error */ 

  do{
    /* perform preprocess semantic function */
    if (grammarCtxt->child &&
      ( errStat = preOrPostProcess(grammarCtxt->child, NULL, '-') ) 
        >= 100) return(errStat);

    /* Process children */
    if (grammarCtxt->child &&
      ( errStat = traverse_solo(grammarCtxt->child)) >= 100 )
	return(errStat);

    /* perform postprocess semantic function  */
    if (grammarCtxt->child &&
      (errStat = preOrPostProcess(grammarCtxt->child, NULL, '+')) >= 100)
      return(errStat);
    } while (grammarCtxt = grammarCtxt->sibling);
  return(errStat);
  }

/**************************************************************************
 * NAME: huntForMoreTags
 *
 * DESCRIPTION: Returns a pointer to the context in the data tree whose tag
 *              matches the current line in the grammar tree. Returns
 *              (NODE *)NULL if no match. The names need to be switched.
 *
 * CALLED BY:   ged_traverse_grammar
 **************************************************************************/
NODE *huntForMoreTags(dataCtxt, grammarCtxt)
  NODE *dataCtxt, *grammarCtxt;
  {
  for (grammarCtxt = grammarCtxt->sibling; grammarCtxt; grammarCtxt = grammarCtxt->sibling)
    if (!ged_match_tags(grammarCtxt, dataCtxt))
      return(grammarCtxt);
  return((NODE *)NULL);  /* no match */
  }

/**************************************************************************
 *
 * NAME: ged_traverse
 *
 * DESCRIPTION: recursive function that traverses the grammar and data trees.
 *              dataCtxt and grammarCtxt are the current levels of the
 *              respective trees.
 *
 * CALLED BY:    ged_process
 * 
 **************************************************************************/

int ged_traverse(dataCtxt, grammarCtxt)
    NODE *dataCtxt, /* Current node of data tree */
         *grammarCtxt; /* Current node of grammar tree */ 
  {
  NODE *currentGramCtxt; /* The tag in the grammar's sibling list that matches
                          * the dataCtxt tag */
  int errStat= DFLT_ERR; /* Status code returned by user-defined functions &
                          * ged_error */ 

  /* See if tag is valid */
  do{
    /*Search fo the tag in grammar's sibling list that matches the dataCtxt tag*/
    if (currentGramCtxt = huntForTags(dataCtxt,grammarCtxt))
      {
      /* perform preprocess semantic function  */
      if (currentGramCtxt->child &&
         (errStat = preOrPostProcess(currentGramCtxt->child,dataCtxt,'-')) >= 100)
        return(errStat);

      /* Process children */
      if (dataCtxt->child && (errStat = ged_traverse(dataCtxt->child,currentGramCtxt)) >=100)
        return(errStat);

      /* perform postprocess semantic function  */
      if (currentGramCtxt->child && (errStat = preOrPostProcess(currentGramCtxt->child,
          dataCtxt,'+')) >= 100)
        return(errStat);
      }
    else 
      {
      if ((errStat = ged_error(301, dataCtxt)) >= 100) /*Skips children ged_error: Tag Not Found*/
        break;
      }
    } while (dataCtxt = dataCtxt->sibling);
  return(errStat);
  }
/**************************************************************************
 * NAME: huntForTags
 *
 * DESCRIPTION: Returns a pointer to the context in the grammar whose tag
 *              matches the current line in the GEDCOM tree. Returns
 *              (NODE *)NULL if no match.
 *
 *              grammarCtxt is the current line in the grammar.
 *              dataCtxt is the current line in the transmission file that
 *              needs to be matched.
 *
 * CALLED BY:   process
 **************************************************************************/

NODE *huntForTags(dataCtxt, grammarCtxt)
 NODE *dataCtxt, *grammarCtxt;
  {
  for (grammarCtxt = grammarCtxt->child; grammarCtxt; grammarCtxt = grammarCtxt->sibling)
    {  
    if (!ged_match_tags(grammarCtxt, dataCtxt))
      return(grammarCtxt);
    }
  return((NODE *)NULL);  /* no match */
  }

/**************************************************************************
 * NAME:  preOrPostProcess
 *
 * DESCRIPTION: Performs the pre or post processing function, if any.
 *              Example Grammar File - Two Types of Records (indentation added).
 *
 *                          0 INDI 1 2
 *                            1 - preFunc
 *                            1 + postFunc 1 3
 *                            1 NAME
 *                            1 BIRT
 *
 * The above grammar instructs gedproc to call two functions each time the
 * tag "INDI" is scanned in the GEDCOM data. The two instruction lines that
 * call these functions must be positioned immediately following the "INDI"
 * tag and must be one level lower.  Notice the tag and value fields of these 
 * two  instruction lines. The value field corresponds to the name of the 
 * function that is called and the plus (+) and minus (-) signs in the tag
 * field determine when the function is called. The minus sign instructs 
 * gedproc to call a function immediately after scanning "INDI." The plus sign
 * tells gedproc not to call the function until after all of the children 
 * of "INDI" have been scanned. This pre and post processing ability is 
 * provided to allow initialization before processing subordinate lines and 
 * analysis after.
 *
 * CALLED BY:  process.
 * CALLS:      ged_get_value, dataCtxt
 * 
 **************************************************************************/

int preOrPostProcess(grammarCtxt, dataCtxt, funcTag)
    NODE *grammarCtxt;
    NODE *dataCtxt;
    char funcTag;
  {
  int errStat = DFLT_ERR;
  int i;
  char *value, *tag;

  /* all + and - tags must be a beginning of sibling list */
  while ((*(tag = ged_get_tag(grammarCtxt))=='+') || *tag == '-')
    {
    if (*tag ==funcTag) /* funcTag & *tag will be a '+' or '-'*/
      {      
    /* if the right function was found, call it. If it is less than numFuncsUsed
     * then the right function was found. */
        
      /* checks for valid function */
      if ((i = get_func(value = ged_get_value(grammarCtxt))) < numFuncsUsed)
        { /* call the function */
        errStat = (*funcsArray[i].funcPointer)(dataCtxt, value);
        if (errStat == -1) /* Exit if -1 */
          exit(errStat); /* Clear the block. It's gonna blow!*/
        if (errStat >=100)
          return(errStat);
        }

      /* did not find function */
      else if ((errStat = ged_error(303, dataCtxt)) >=100 )
	return(errStat); /* ged_error: Hook not found */
      }

    if (!(grammarCtxt = grammarCtxt->sibling))
      break; /* stop if no more siblings on this level */
    }
    return(errStat);
  }

/**************************************************************************
 * NAME:  ged_define_semantic
 *
 * DESCRIPTION:  Function registration.  This will register the semantic
 *               functions created by the user.
 * 
 * CALLED BY:    user
 **************************************************************************/

int ged_define_semantic(funcName, funcPointer)
  byte *funcName; /* Name of user's semantic function */
  #ifdef CYBER
  int (*funcPointer)();  /* pointer to the function to call */
  #else
  int(*funcPointer)(NODE *, byte *);  /* pointer to the function to call */
  #endif
        
  {
  /* perform any needed error checking */
  if (strlen(funcName)>=FUNC_NAME_SIZ )
    return(2); /* Function name too large */
  if (numFuncsUsed >= MAX_FNCS)
    return(3); 

  /**Inititialize the function **/
  strcpy(funcsArray[numFuncsUsed ].name, funcName);
  funcsArray[numFuncsUsed++].funcPointer = funcPointer;
  }

/**************************************************************************
 * NAME:  get_func
 *
 * DESCRIPTION:  pass a pointer to the value line of a tag and this will
 *               call the corresponding function.
 *
 * RETURN VALUE: Returns the position in the function array.  If this position
 *               is equal to numFuncsUsed, then the right function was not
 *               found.
 *               
 * CALLED BY:    preProcess, postProcess
 **************************************************************************/
 int get_func(line)
    char *line;
  {
  int i;
  for (i=0; i < numFuncsUsed; i++)
    if (!ged_match(line, funcsArray[i].name))
      return(i);
  return(i);
  }

/**************************************************************************
 * NAME: ged_load_grammar
 *
 * DESCRIPTION:  loads the grammar into memory.  The pool used for this
 * grammar lives through the whole program.  The siblings that are normally
 * at level 0 are all changed to level 1 by creating a transparent parent
 * for them. Currently GLIMIT is defined as (unsigned long) 2500.  This
 * will be changed in the future to be user-defined.
 *
 * CALLED BY:  main
 **************************************************************************/

NODE *ged_load_grammar(fg)
  FILE *fg;

  {
  NODE *top, *node;
  short status = MORE_RECORDS;

  /* create a level zero root */
  top = ged_copy_node("grammar", (NODE *)NULL, (NODE *)NULL);
  while (status == MORE_RECORDS )
    {
    node = read_tree(fg, &status, GLIMIT,0,NULL,0); /* 0 set to read normal 0 records */
    if ((status==END_OF_FILE) || (node == (NODE *)NULL))
      break;
    ged_connect_child(top, node, -1);
    }
  return(top);
  }

/**************************************************************************
 *
 * NAME:  call_grammar_functions
 *
 * DESCRIPTION:  this function is called by ged_apply_tree whenever their is
 * a DEFINITION node in the GRAMMER.  THEN this function intern calls the
 * functions from the node as specified by the grammar lines within the
 * Definition tree.
 * CALLED BY:    ged_apply_tree from within the ged processor
 *
 **************************************************************************/
int call_grammar_functions(int level, NODE *defintnTree, int arg1, int arg2)
  {
   int i;
   int errStat = DFLT_ERR;
   char funcName[31];
   char *funcArg, *funcNamePtr;

    funcNamePtr = ged_get_tag(defintnTree);
    strncpy(funcName, funcNamePtr, 30);
    ged_clip_word(funcName);
    funcArg =  ged_get_value(defintnTree);
    if ((strncmp(funcName, "DEFINITION", 10)) == 0)
       return(DFLT_ERR);
    i = get_func(funcName);
    if (i < numFuncsUsed )
      {
       errStat = (*funcsArray[i].funcPointer)(defintnTree, funcArg);
      if (errStat == -1) /* exit if -1 */
	 exit(errStat);
      if (errStat >= 100)
	return(errStat);
      }
    return(errStat);
   }

