/* testpic.c
Program to generate a test "picture" coded for Tektronix 4010 graphics,
especially for testing animation program ANIMATE.

Tek4010 coordinates have 0 <= x <= 1023 and 0 <= y <= 779
with (x,y)=(0,0) lying in the lower left corner of the screen.
ASCII character GS = 29 (base 10) = 1d (base 16) turns on graphics mode.
Each coordinate pair (x,y) is coded as 4 bytes;
see xy_to_Tek4010() below for details.

With Tek4010 graphics, a curve is drawn by sending GS to signal the beginning
of a curve followed by 4-bytes for each (x,y) coordinate pair
that defines the curve.

ANIMATE assumes that a graph is done when ANIMATE encounters ESCape FormFeed
(ESC FF) or is switched into alphanumeric mode by receiving
either ASCII character CR = 0d (base 16) or US = 1f (base 16).
The interpretation of US will be changed in the next edition of
ANIMATE so that it will behave according to the Tektronix standard.

Jon Ahlquist, 3 April 1990.
Dept of Meteorology B-161
Florida State University
Tallahassee, FL 32306-3034
Telephone: (904) 644-1558
Internet:  ahlquist@metsat.met.fsu.edu (ahlquist@128.186.5.2)
*/

/* Declare function prototypes. */
#include <process.h> /* exit()  */
#include <stdio.h>   /* fopen(), fputc(), fputs(), fclose(). */
void fput4b(char *byte, FILE *fp);
int xy_to_Tek4010(int ix, int iy, char *byte);

/* Define a few ASCII constants.       */
#define  GS 0x1d /* Graphics Shift     */
#define  CR 0x0d /* Carriage Return    */
#define  LF 0x0a /* Line Feed          */
#define  US 0x1f /* control Underscore */
#define ESC 0x1b /* ESCape             */
#define  FF 0x0c /* Form Feed          */

/* Meaning of the ASCII constants for Tek 4010 operation,
especially with ANIMATE.

GS     turn on graphics mode (if not already invoked) and start
       drawing a curve (as a set of connected line segments).
CR     turn on alphanumeric mode.
       For ANIMATE, this signals the end of a plot.  See exception below.
CR LF  should be ignored by ANIMATE.
       A commercial Tek4010 translator I encountered inserted CR LF
       every so often even when alphanumeric mode was not being invoked.
       Therefore, ANIMATE is written to ignore CR LF.
US     turn on alphanumeric mode without changing the current position
       of the spot of light on the screen.
       This can be used to label graphs.
       For the current version of ANIMATE, this signals the end of a plot.
ESC FF clear the graphics screen.
       For ANIMATE, this signals the end of a plot.
*/

void main(void)
{

FILE *fp;
char  byte[4];

if ((fp = fopen("testpic.tek", "wb")) == NULL)
   {
   printf("Could not open output file.\n");
   exit(1);
   }

/* Draw a right angle on the screen, but include some extraneous
junk to test ANIMATE's ability to handle it. */

/* Start the file with junk consisting of alphanumeric information,
including CR LF. ANIMATE should ignore everything before it encounters
ASCII character GS, which turns on the graphics mode. */
fputs("abc", fp);
fputc(CR,   fp);
fputc(LF,   fp);

/* Turn on graphics mode and signal the start of a "curve." */
fputc(GS, fp);

/* Draw a horizontal line segment. */
xy_to_Tek4010(  0, 100, byte);
fput4b(byte, fp);
xy_to_Tek4010(500, 100, byte);
fput4b(byte, fp);

/* Insert an extraneous carriage return and line feed. */
fputc(CR, fp);
fputc(LF, fp);

/* Draw a vertical line segment by moving upward from the previous point. */
xy_to_Tek4010(500, 700, byte);
fput4b(byte, fp);

fclose(fp);
}
/*************************************************************/
void fput4b(char *byte, FILE *fp)
/* Write 4 bytes to a file stream. */
{
int i;
for (i=0; i<4; i++) fputc(*(byte+i), fp);
}

/*************************************************************/
int xy_to_Tek4010(int ix, int iy, char *byte)
/* This function converts an (ix,iy) coordinate on a Tektronix 4010
graphics screen into the 4 byte sequence that is used in the Tek4010
graphics language to code that coordinate pair.
If either ix or iy lies out of range (0 through 1023 for ix,
0 through 779 for iy), control returns to the calling
program without altering the values in the character array "byte,"
and the integer return value for xy_to_Tek4010 is set equal to 1.
If both ix and iy lie in range, the integer return value for
xy_to_Tek4010 is set equal to 0 and the four byte sequence representing
ix and iy in Tek4010 graphics language is loaded into character array "byte."
This sequence has the form and order: 001YYYYY 011yyyyy 001XXXXX 010xxxxx,
where upper case letters denote the leftmost  5 bits of a coordinate,
and   lower case letters denote the rightmost 5 bits of a coordinate.

Jon Ahlquist, 28 Feb 1990.
*/
{
/* Return if ix or iy lies out of range. */
if ((ix<0) || (ix>1023)) return(1);
if ((iy<0) || (iy> 779)) return(1);

/* Store Tek4010 coordinates ix and iy as an ordered 4 byte string.
The first 3 bits of each byte are a prefix that tells what kind
of a coordinate is represented by the final 5 bits.
The same prefix is used for the high 5 bits of x and y. */
*byte     = 32 + (iy >> 5); /* Bits 001 followed by high 5 bits of iy. */
*(byte+1) = 96 + (iy % 32); /* Bits 011 followed by low  5 bits of iy. */
*(byte+2) = 32 + (ix >> 5); /* Bits 001 followed by high 5 bits of ix. */
*(byte+3) = 64 + (ix % 32); /* Bits 010 followed by low  5 bits of ix. */
return(0);
}
