	title	BCDASM -- Copyright 1997, Morten Elling
	subttl	Perform a digit right-shift (DIV 10^n) on a packed BCD

	include	model.inc
	include	modelt.inc
	include	bcd.ash

	@CODESEG

;//////////////////////////////////////////////////////////////////////
;//	Name	bcdShr
;//	Desc	Perform a digit right-shift of a packed signed BCD.
;//
;//
;//	Entry	Passed args
;//	Exit	Packed signed BCD returned to destination.
;//		Accumulator (and zf flag) determined:
;//		Acc = 0: Result is zero
;//		Acc > 0: Result is non-zero
;//		(carry, sign, and overflow flags should be ignored)
;//
;//	Note	A 10-byte BCD holds 18 BCD digits ( 2*(10-1) ).
;//
;//	Note	This procedure shifts one digit, i.e. one nibble, per
;//		shift count effectively performing a
;//		DIV 10-to-the-n'th-power operation.
;//		Zeros are shifted into the high-order of destination.
;//		The sign of the BCD is changed only if the result is
;//		zero. If the shift count is larger than or equals the
;//		number of digits in the BCD, the result will be zero.

bcdShr	proc
arg	dstBCD	:dataptr, \	; Addr of destination BCD
	dstsz	:@uint, \	; Byte size of BCD
	count	:@uint		; No. of BCD digits to shift
@uses	ds,es,rsi,rdi,rbx,rcx,rdx
;.
; ----- Load registers
	@cld			; String ops forward
	@LDS  rsi, [dstBCD]
	@LES  rdi, [dstBCD]
	mov   rdx, [count]
	mov   rbx, [dstsz]
	dec   rbx		; # bytes, excl. sign byte

; ----- Handle special cases
	test  rdx, rdx 		; If under-sized shift,
	jz sh @@chkz		;   see if zero
	mov   rcx, rdx		; Shift count
	shr   rcx, 1		; Make # bytes
	jz sh @@odd		; If zero, shift count = 1
	cmp   rcx, rbx		; If over-sized shift,
	jae sh @@clear		;   zero-fill

; ----- Shift the even digit pairs
	add   rsi, rcx		; Point to source
	mov   rax, rcx		; Save count/2
	neg   rcx		; - Count/2
	add   rcx, rbx		; + Field width = # bytes to shift
	rep   movsb		; Shift towards LSB (rsi > rdi)
	mov   rcx, rax		; Count/2 = # bytes to zero
	sub   al, al		; Zero al
	rep   stosb		; Zero high-order of number

	sub   rdi, rbx		; Point to LSB
	shr   rdx, 1		; Is there an odd digit?
	jnc sh @@chkz	 	; No, see if result is zero

	mov   rsi, rdi		; Point to LSB
	mov   rcx, rdx		; Get # bytes shifted
	neg   rcx		; Negate
@@odd:	add   rcx, rbx		; # bytes to 'odd-shift'

; ----- Shift the odd digit
	if @isUse32
	sub   rax, rax		; Clear high(eax)
	endif
	mov   rdx, rdi		; Save LSB pointer
@@rt:	lodsb			; Get a byte
	mov   ah, [rsi] 	; Get the next byte
	@shr  rax, 4		; Shift acc right one nibble
	stosb			; Store al to destination
	loop  @@rt		; Loop until done
	mov   rdi, rdx		; rdi -> LSB

; ----- Check if result is zero
@@chkz:	; rdi -> LSB
	mov   rcx, rbx		; REP count
	sub   al, al		; Zero al
	repz  scasb		; See if all zeros (cf=0 if zero)
	rcr   al, 1		; Set to 80h if result non-zero
	add   rdi, rcx		; Point to sign byte
	jmp sh @@sign

; ----- Shift count > # digits: zero-fill
@@clear: ; rdi -> LSB
	mov   rcx, rbx		; REP count
	sub   al, al		; Zero al
	rep   stosb		; Whip it

; ----- Determine sign and return value
@@sign:	; rdi -> sign byte
	; al = 0 if result zero, else 80h
	and   @ES [rdi], al	; Clear sign bit if result zero
	rol   al, 1		; Return acc 0 or 1
	and   rax, 1		; Determine acc and zero flag
	RET
bcdShr	endp

	END