/*****************************************************************************
*
* Base to base conversion DLL. Version 1.0 Beta
* Copyright (C) 1995 Sammy Yousef
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation.
*
* This DLL is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
* This library provides routines for conversion of numbers between bases.
* Storing the number as a double floating point does cause inaccuracies in
* the translations. It is intended for use in Windows which is a trademark
* of the Microsoft Corporation. This DLL received only minimal testing.
*
* This library was written for readability and ease of use NOT efficiency
*
* *##################*Use this DLL at your own risk.*###################*
*
* If you use the DLL and have any comments feel free to email me. 
* I would be most grateful for any feedback about either the quality of the
* code or the reliability of these routines.
d*
* Author : Sammy Yousef
*              syousef@socs.uts.edu.au
*              syousef@ghostgum.itd.edu.au
*              sammy@ftoomsh.socs.uts.edu.au
*
*          Also known as Whiteheart on the talkers.
*
* Date   : 2 August 1995
*
*****************************************************************************/

#include "btbdll.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <math.h>
#include <windows.h>

/**********************************************************************
* Routines to convert between bases (these routines are split into 2
* sections - see the following function for an explanation of why)
***********************************************************************/

    /**********************************************************************
    * Give the decimal integer value of a digit
    * Returns -1 for invalid
    * (moved to top due to a problem in
    * the compiler thinking this function was redeclared incorrectly if used
    * before declared)
    ***********************************************************************/
    int FAR PASCAL _export decimal_integer(char indigit)
    {
        int charval;
        char upperdigit;

        upperdigit=toupper(indigit);

        if ( (upperdigit >= '0') && (upperdigit <= '9') )
            {
                charval= (int) upperdigit - 48;
            }
        else if ( (upperdigit >= 'A') && (upperdigit <= 'Z') )
            {
                charval= (int) upperdigit - 55;
            }
        else
            {
                charval=-1;
            }

        return(charval);
    }

/**********************************************************************
* Routines to check for valid input
***********************************************************************/

    /**********************************************************************
    * Check a single character is valid for use as a digit given the base
    ***********************************************************************/
    int FAR PASCAL _export check_digit(int inbase, char indigit)
    {
        int checkint;

        checkint = decimal_integer(indigit);

        if ( ( checkint >= inbase ) || ( checkint < 0 ) )
            {
                fprintf(stderr,"\nDigits in Input number must be between zero "
                                "and (Input base - 1).\n");
                return(1);
            }
        else
            {
                return(0);
            }
    }

    /**********************************************************************
    * Check a string is valid for use as a set of digit given the base
    ***********************************************************************/
    int FAR PASCAL _export check_numstring(int inbase, char numstring[MAXSTRINGIN])
    {
        int counter;
        int radix_count;
        int this_digit_ok;
        int numstringlen;
        numstringlen=strlen(numstring);

        radix_count=0;

        for ( counter=0 ; counter < numstringlen ; counter++ )
            {
                if ( ( counter == 0 ) && ( ( numstring[counter] == '+' ) || ( numstring[counter] == '-' ) ) )
                    {  /* Check to see if it is a +/- sign at the front of the number */
                    }  /* Continue on if this is the case */
                else
                    {
                        if ( numstring[counter] == '.' )
                            {
                                if ( radix_count > 1 )
                                    {
                                        fprintf(stderr,"There are too many radix points in this input value.\n");
                                    }
                                else
                                    {
                                        /* if the number is the radix point and no radix point has yet been */
                                        /* encountered, simply increment the radix_count and proceed */
                                        radix_count++;
                                    }
                            }
                        else
                            {
                                this_digit_ok=check_digit(inbase, numstring[counter]);
                                if ( this_digit_ok == 1 )
                                {
                                    fprintf(stderr,"Invalid digit(s).\n");
                                    return(1);
                                }
                            }
                    }        
            }

        return(0);
    }

    /**********************************************************************
    *Check the bases for input and output are valid
    ***********************************************************************/
    int FAR PASCAL _export check_bases(int inbase,int outbase)
    {
        if ( (inbase < 2 ) || (inbase > MAXBASE) )
            {
                fprintf(stderr,"Invalid base supplied for Input Number.\n");
                return(1);
            }

        if ( (outbase < 2) || (outbase > MAXBASE) )
            {
                 fprintf(stderr,"Invalid base supplied for Output Number.\n");
                 return(1);
            }

        return(0);
    }

    /**********************************************************************
    * make sure there is at most 1 radix point
    ***********************************************************************/
    int FAR PASCAL _export check_radix(char innumber[MAXSTRINGIN])
    {
        int radixcount;
        int loopcount;
        int innumber_length;
   
        radixcount=0;
   
        innumber_length=strlen(innumber);
   
        for ( loopcount=0 ; loopcount < innumber_length ; loopcount++ )
            {
                if ( innumber[loopcount] == '.' )
                    {
                        radixcount++;
                    }
            }
   
        if ( radixcount > 1  )
            {
                fprintf(stderr,"\nThere are too many radix points in this input value.\n");
                return(1);       /* if there is more than one '.' in the string */
            }                    /* the user input is incorrect */

        return(0);
    }

/**********************************************************************
* Routines to convert between bases (continued)
***********************************************************************/
    /**********************************************************************
    * Give the base x character corresponding integer value of a digit
    * Returns # for invalid
    ***********************************************************************/
    char FAR PASCAL _export character_integer(int indigit)
    {
        char outchar;
        
        if ( indigit >= 0 && indigit <= 9 )
            {
                outchar = (char) indigit + 48;
            }
        else if ( (indigit >= 10) && (indigit <= 35) ) 
            {
                outchar = (char) indigit + 55;
            }
        else
            {
                outchar = '#';
            }
         
         return(outchar);
    }

    /**********************************************************************
    * Convert an input string from an input base to base 10
    ***********************************************************************/
    double FAR PASCAL _export whole_to_ten(int inbase,char innumber[MAXSTRINGIN])
    {
        int    position;
        double baseten=0;
        int    length;
        int    power;
        int    foreign_digit;

        length=strlen(innumber);
        for ( position = length ; position > 0 ; position-- )
        {
            
            /* handle characters + and - */
            if ( ( position == 1 ) && ( innumber[ ( position - 1 ) ] == '+') )
                {
                }
            else if ( ( position == 1 ) && ( innumber[ ( position - 1 ) ] == '-') )
            {
                /* make number negative if first character is '-' 
                   note that we subtract - 0.1 - the idea is that this will
                   not change the whole number value of the number in base 
                   10 but if the number is zero will still ensure that it 
                   remains negative. This is essentially a kludge and may
                   cause problems. I included this reluctantly because initially
                   this library was written without taking sign into account.
                   The alternative would have been to add another parameter
                   to the set of routines which would have been messy and taken
                   time to test properly.
                */
                
                baseten = - baseten - 0.1;
            }    

            else /* not a + or - sign at the first position */
            {
                power = length - position;
                foreign_digit = decimal_integer(innumber[ ( position - 1 ) ]);
                baseten = baseten + foreign_digit * pow(inbase,power);
            }    
        }
        return(baseten);
    }
    
    /**********************************************************************
    * Convert an input "fraction" string from an input base to base 10
    ***********************************************************************/
    double FAR PASCAL _export fraction_to_ten(int  inbase,
    					      char innumber[MAXSTRINGIN])
    {
        int    position;
        double baseten=0;
        int    length;
        int    power;
        double    foreign_digit;

        length=strlen(innumber);
        for ( position = 1 ; position <= length ; position++ )
        {
            power = -position;
            foreign_digit = decimal_integer(innumber[( position - 1)]);
            baseten = baseten + foreign_digit * pow(inbase,power);
        }
        return(baseten);
    }
    
    /**********************************************************************
    * Convert whole number part of an input base 10 double 
    * to base outbase - result in output_string
    ***********************************************************************/
    void FAR PASCAL _export baseten_to_basex_whole(
			    char output_string[MAXSTRINGOUT],
                           int outbase,double base10)
    {
        double dividend;
        double quotient;
        double floored_quotient;
        double fract_remainder;
        int    integer_remainder;
        int    make_negative;
        char   character_remainder;
        char   temp_string[MAXSTRINGOUT];
        
        make_negative = 0;
        dividend = base10;

        strcpy(output_string,"");
        
        if ( dividend < 0 )
            {
                /* If the number is negative flag for output of a minus sign
                   at the beginning of the number then treat the number
                   as if it were positive
                */
                   
                dividend = -dividend;
                make_negative=1;
            } 
        else if ( dividend == 0 )
                {
                    sprintf(output_string,"%c",'0');
                }
            
        while ( dividend > 0 ) 
            {
                quotient = dividend / outbase;
                floored_quotient = floor(quotient);
               
                fract_remainder = 
                    ( dividend - ( floored_quotient * outbase ) );
                integer_remainder = floor(fract_remainder);

                character_remainder = character_integer(integer_remainder);
                
                /* append the digit to the front of the number */
                sprintf(temp_string,"%c%s",character_remainder,output_string);
                sprintf(output_string,"%s",temp_string);

                dividend = floored_quotient;
            }

       /* Put a negative sign at the front of the number */
       if ( make_negative == 1 )
           {
                /* append a minus sign to the front of the number if input
                   was negative
                */
                sprintf(temp_string,"%c%s",'-',output_string);
                sprintf(output_string,"%s",temp_string);
           }     
    }
    
    /**********************************************************************
    * Convert fractional part of an input base 10 double 
    * to base outbase - result fraction digits in output_string 
    ***********************************************************************/
    void  FAR PASCAL _export baseten_to_basex_fraction(
    			char output_string[MAXSTRINGIN],
                      	int outbase,double base10)
    {
        double fraction_left;
        int    digit;
        char   chardigit;
        int    deccount = 0;
        char   number_string[MAXSTRINGOUT];
        
        strcpy(output_string,"");
        
        fraction_left = base10; 

        while ( (fraction_left != 0) && (deccount <= MAXDECIMAL ))
           {
               fraction_left = fraction_left * outbase;
               digit = floor(fraction_left);
               
               sprintf(number_string,"%c",character_integer( digit ));
               strcat(output_string, number_string);
               
               fraction_left = fraction_left - digit;
               deccount = deccount + 1;
           }

    }

    /**********************************************************************
    * split the number into two parts. these are the parts either side of
    * the radix point
    ***********************************************************************/
    int  FAR PASCAL _export split_at_radix( char innumber[MAXSTRINGIN],
        		                    char left_of_radix[MAXSTRINGIN],
		                            char right_of_radix[MAXSTRINGIN])
    {
        /* NOTE: strtok was used here orignially providing a compact and simple
        solution to splitting at the radix. However compiling the same code that
        worked in DOS using Borland C++ 4.02 under LINUX with GCC caused this
        function to return rubbish for the second arguement. This solution appears
        to work in both environments */

        char* decimal_point;
        
        if ( check_radix(innumber) == 1 ) 
            return(1);           /* make sure there is at most 1 radix point */ 

        strcpy(right_of_radix,"");
        strcpy(left_of_radix,innumber);
        decimal_point = strchr(left_of_radix,'.');
        
        if ( decimal_point != NULL )
            {
    	        strcpy( right_of_radix, decimal_point + sizeof(char));
    	        *decimal_point = NULL;
            }
       
        return(0);
    }

    /**********************************************************************
    * Recombine the whole and fractional parts of the number
    ***********************************************************************/
    void FAR PASCAL _export recombine(char complete_number[MAXSTRINGOUT],
        		              char left_hand_side[MAXSTRINGOUT],
		                      char right_hand_side[MAXSTRINGOUT])
    {
        strcpy(complete_number, left_hand_side);
        
        if ( strlen(right_hand_side) != 0 )
            {
               strcat(complete_number, ".");
               strcat(complete_number, right_hand_side);
            }
    }
    
    /**********************************************************************
    * takes the original base followed by the output base then
    * the number as a string expressed in terms of the input base
    * the function then changes the fourth parameter to the converted
    * number.
    ***********************************************************************/
    int FAR PASCAL _export btb(int inbase, int outbase, char innumber[MAXSTRINGIN], char outnumber[MAXSTRINGOUT])
    {
        double baseten_left_of_radix;           /* intermediate base 10 doubles */
        double baseten_right_of_radix;

        char left_of_radix[MAXSTRINGIN];        /* these become the split numbers to the */
        char right_of_radix[MAXSTRINGIN];       /* right and left of the radix point     */
                                                /* respectively for the INPUT number     */

        char basex_left_of_radix[MAXSTRINGIN];  /* these become the split numbers to the */
        char basex_right_of_radix[MAXSTRINGIN]; /* right and left of the radix point     */
                                                /* respectively for the OUTPUT number    */

        if ( check_bases(inbase,outbase) == 1 ) /* check that both input and output bases */
            return(1);                          /* are valid bases.  */

        if (check_numstring(inbase,innumber) == 1 )
            return(1);


        /* Split number into whole and fraction*/
        if ( split_at_radix(innumber,left_of_radix,right_of_radix) == 1 )
                return(1);

        /* convert whole and fraction to double precision base 10 numbers */
        baseten_left_of_radix = whole_to_ten( inbase, left_of_radix);
        baseten_right_of_radix = fraction_to_ten(inbase, right_of_radix);


        /* convert double precision base 10 to basex string */
        baseten_to_basex_whole(basex_left_of_radix, outbase,baseten_left_of_radix);
        baseten_to_basex_fraction(basex_right_of_radix, outbase, baseten_right_of_radix);

        /* recombine and copy the converted numbers to the output number */
        recombine(outnumber, basex_left_of_radix, basex_right_of_radix);
        
        return(0);
    }

    /**********************************************************************
    * LibMain as required for all DLL's
    ***********************************************************************/
    int FAR PASCAL LibMain(HINSTANCE hinstance, WORD wdataseg,
    			   WORD word, LPSTR lpstr)
    {
    }


