#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <conio.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <errno.h>
#include <assert.h>
#include <dos.h>
#include <dir.h>
#include <math.h>

/*
  i think this is the technically correct for TRUE & FALSE
*/
#define FALSE (0)
#define TRUE  (!FALSE)

/*
  redefine this if you want a different switch character
  (unix-ish would be '-')
*/
#define SWITCHCHAR '/'

/*
  this is a useful macro for internal structures
*/
#define STRUCTCOUNT(x) (sizeof(x)/sizeof(*x))

/*
  this makes more sense to me
  LOCAL  is for functions only visible to this file
  GLOBAL is for functions visible to the world

  note that LOCAL is only used for file-level stuff. static
  is still used inside functions.
*/
#define LOCAL static
#define GLOBAL

/*
  file actions
*/
typedef enum {
  ACTION_NULL = 0,
  ACTION_ADD,
  ACTION_UPDATE,
  ACTION_DELETE
} FILEACTION;
/*
  program option flags
*/
#define FLAG_RECURSE  0x0001
#define FLAG_FORCEADD 0x0002
#define FLAG_DELETE   0x0004
#define FLAG_DELDIR   0x0008
#define FLAG_ADD      0x0010
#define FLAG_UPDATE   0x0020
#define FLAG_BOTH     0x0040
#define FLAG_VERBOSE  0x0080
#define FLAG_NOACTION 0x0100
#define FLAG_FORCEUPD 0x0200
#define FLAG_FORCEDEL 0x0400

#define FLAG_FORCEALL (FLAG_FORCEADD | FLAG_FORCEDEL | FLAG_FORCEUPD)

/*
  file info that i want to maintain
*/
typedef struct tagFILEINFO {
  char      *fullName;     /* full name to file (- base path) */
  unsigned   time;         /* misc. file info */
  unsigned   date;
  long       size;
  char       attr;
  FILEACTION action;       /* action to take */
} FILEINFO;


/*
  file lists that i want to maintain
*/
typedef struct tagFILELIST {
  char      *basePath;  /* base path. this is removed from all files */
                        /* in [info] */
  FILEINFO  *info;      /* array of files + info */
  int        ct;        /* number of files in list */
  int        addCt;     /* number of files to be added */
  int        updCt;     /* number of files to be updated */
  int        delCt;     /* number of files to delete */
  long       addBytes;  /* total number of bytes to be added   */
  long       updBytes;  /* total number of bytes to be updated */
  long	     delBytes;  /* total number of bytes to be deleted */
} FILELIST;

/*
  misc. stuff
*/
LOCAL FILELIST srcFiles;
LOCAL FILELIST dstFiles;
LOCAL char     includeFiles[1024];
LOCAL char     excludeFiles[1024];
LOCAL unsigned flags;
LOCAL int      outOfSpace;
LOCAL FILE    *logFile;
LOCAL long     addFile,  updFile,  delFile;
LOCAL long     addBytes, updBytes, delBytes;
LOCAL long     startTime;

LOCAL long     toCopy;      /* total process stats */

LOCAL char    *scrollList[10];
LOCAL int      scrollPos;

LOCAL int      fileCt;

LOCAL int      verboseScreen; /* this is set when InitVerboseScreen is called */
LOCAL int      finishAndAbort; /* this is set for ``abort when finished'' */

/**********************************************************
  i use these memory routines so i don't have to deal
  with out of memory errors in the rest of the program
  (they're always fatal)
 **********************************************************/

#define MEM_ERROR_NULL       0
#define MEM_ERROR_CAPACITY   1
#define MEM_ERROR_OUT_OF_MEM 2

LOCAL void memError(int err)
{
  char *errMsg[] = {
    "No Error",
    "Maximum allocation exceeded",
    "Out of conventional memory"
  };
  if ((err < 0) || (err > STRUCTCOUNT(errMsg)))
    fprintf(stderr, "\n\nInternal Error\n\n");
  else
    fprintf(stderr, "\n\n%s\n\n", errMsg[err]);
  exit(1);
}

LOCAL void *chkMalloc(long sz)
{
  void *ptr;
  if (sz > 65500)
    memError(MEM_ERROR_CAPACITY);
  else if ((ptr = malloc(sz)) == 0)
    memError(MEM_ERROR_OUT_OF_MEM);
  return ptr;
}

LOCAL void *chkRealloc(void *base, long sz)
{
  if (!base)
    return chkMalloc(sz);
  else if (sz > 65500)
    memError(MEM_ERROR_CAPACITY);
  else if ((base = realloc(base, sz)) == 0)
    memError(MEM_ERROR_OUT_OF_MEM);
  return base;
}

LOCAL void chkFree(void *ptr)
{
  if (ptr)
    free(ptr);
}

LOCAL char *strdup(const char *s)
{
  return strcpy(chkMalloc(strlen(s)+1), s);
}

/**********************************************************
  misc. screen output stuff
 **********************************************************/

/*
  show elapsed time hh:mm:ss
*/
LOCAL void ShowElapsed(int x, int y, unsigned elapsed)
{
  gotoxy(x,y);
  cprintf("%2d:%02d:%02d",
          (int) (elapsed/3600),
          (int) ((elapsed/60) % 60),
          (int) (elapsed % 60));
}

/*
  show progress bar
  pct = percentage*10 (0..1000)
*/
LOCAL void ShowProgress(int x, int y, unsigned pct)
{
  #define PLEN (67L)
  static char scrbuf[PLEN+1];

  int ii;
  char buf[4];
  memset(scrbuf, ' ', PLEN);
  scrbuf[PLEN] = 0;
  itoa(pct/10, buf, 10);
  pct = (pct * PLEN / 1000);
  gotoxy(x,y);
  for (ii=0; ii < PLEN; ii++)
    scrbuf[ii] = (ii < pct) ? '' : ' ';
  x = (PLEN-6)/2;
  scrbuf[x] = ' ';
  memcpy(scrbuf+x+1, buf, strlen(buf));
  scrbuf[x+1+strlen(buf)]=' ';
  cputs(scrbuf);

  #undef PLEN
}

/*
  show number amount
  ###,###c
  where c = blank, 'K', 'M'
*/
LOCAL void ShowAmount(int x, int y, unsigned long amt)
{
  char ch;

  #define K (1024UL)
  #define M (K*K)

  gotoxy(x,y);

  if (amt >= 999999999UL) {
    ch = 'M';
    amt = (amt + M/2) >> 20;
  } else if (amt >= 999999UL) {
    ch = 'K';
    amt = (amt + K/2) >> 10;
  } else
    ch = ' ';

  if (amt >= 1000)
    cprintf("%3d,%03d", (int) (amt/1000), (int) (amt % 1000));
  else
    cprintf("    %3d", (int) amt);
  putch(ch);

  #undef M
  #undef K
}

/*
  show or hide the cursor
*/
LOCAL void ShowCursor(int show)
{
  struct REGPACK regs;

  memset(&regs, 0, sizeof(regs));
  regs.r_ax = 0x0300;
  regs.r_bx = 0x0000;
  intr(0x10, &regs);
  if (show)
    regs.r_cx &= ~0x2000;
  else
    regs.r_cx |= 0x2000;
  regs.r_ax = 0x0100;
  intr(0x10, &regs);
}

/*
  update the verbose screen
*/
LOCAL void UpdateVerboseScreen(long amtCopiedFile, long toCopyFile)
{
  long now = clock();
  long amtCopied = toCopy - addBytes - updBytes + amtCopiedFile;
  int ii;

  ShowCursor(0);
  ShowAmount(62, 3, addFile);
  ShowAmount(62, 4, updFile);
  ShowAmount(62, 5, delFile);

  ShowAmount(71, 3, addBytes);
  ShowAmount(71, 4, updBytes);
  ShowAmount(71, 5, delBytes);

  ShowElapsed(29, 7, (now - startTime)/CLK_TCK);
  if (amtCopied)
    ShowElapsed(71, 7, ((float) (now - startTime)) * toCopy / CLK_TCK / max(1, amtCopied) - (now - startTime)/CLK_TCK);

  ShowProgress(11,  9, 1000.0 * max(1, amtCopied) / max(1, toCopy));
  ShowProgress(11, 11, 1000.0 * amtCopiedFile / max(1, toCopyFile));

  for (ii=0; ii < scrollPos; ii++) {
    gotoxy(1, 14+ii);
    cprintf("%-79.79s", scrollList[ii]);
  }

  ShowCursor(1);
}


LOCAL void InterruptProgram(void)
{
  static unsigned *keyBuf = MK_FP(0x40, 0x1a);
  /*
    keyBuf[0] = keyboard buffer head
    keyBuf[1] = keyboard buffer tail

    i do this because it's <<much>> faster than kbhit()
  */
  if ((keyBuf[0] != keyBuf[1]) && (getch() == 0x1b)) {
    int ch;

    /*
      clear any pending keystrokes
    */
    while (kbhit())
      getch();

    cprintf("\r\nInterrupt detected. (C)ontinue, (A)bort, (F)inish?");
    do {
      putch(0x07);
      ch = tolower(getch());
    } while (!strchr("caf", ch));
    cprintf("\r%78c\r", ' ');
    if (ch == 'a')  {
      if (logFile) {
        fprintf(logFile, "<<user interrupt --- last action aborted\n");
        fclose(logFile);
      }
      exit(3);
    }
    finishAndAbort = (ch == 'f');
  }
}

/*
  intialize the verbose screen
*/
LOCAL void InitVerboseScreen(void)
{
  clrscr();
  cputs(
"             source         destination          total           remaining\n\r"
"         files   bytes     files   bytes     files   bytes     files   bytes\n\r"
"add\n\r"
"update\n\r"
"delete\n\r"
"\n\r"
"% Complete          elapsed:                    remaining (estimate):         \n\r"
"         Ŀ\n\r"
"Total:                                                                      \n\r"
"         Ĵ\n\r"
"File:                                                                       \n\r"
"         \n\r"
  );

  ShowAmount(8, 3, srcFiles.addCt);
  ShowAmount(8, 4, srcFiles.updCt);

  ShowAmount(17, 3, srcFiles.addBytes);
  ShowAmount(17, 4, srcFiles.updBytes);

  ShowAmount(26, 3, dstFiles.addCt);
  ShowAmount(26, 4, dstFiles.updCt);
  ShowAmount(26, 5, dstFiles.delCt);

  ShowAmount(35, 3, dstFiles.addBytes);
  ShowAmount(35, 4, dstFiles.updBytes);
  ShowAmount(35, 5, dstFiles.delBytes);

  ShowAmount(44, 3, srcFiles.addCt + dstFiles.addCt);
  ShowAmount(44, 4, srcFiles.updCt + dstFiles.updCt);
  ShowAmount(44, 5, srcFiles.delCt + dstFiles.delCt);

  ShowAmount(53, 3, srcFiles.addBytes + dstFiles.addBytes);
  ShowAmount(53, 4, srcFiles.updBytes + dstFiles.updBytes);
  ShowAmount(53, 5, dstFiles.delBytes);

  UpdateVerboseScreen(0, 0);

  verboseScreen = 1;
}

/*
  return TRUE if [test] is represented by [expr]
  (this is <<not>> case sensitive)
  ...
    the exclude list is a rather standard MSDOS exclude
      '?' matches any single character
      '*' matches any group of zero or more characters
    the difference is this works somewhat more logically, as in
      *\windebug\* will exclude all subdirectories names ``windebug''
*/
LOCAL int isMatch(char *expr, char *test) {
  char *pathPtr   = test,      /* scratch path ptr */
       *exclPtr   = expr,       /* scratch match ptr */
       *lastSrchP = NULL,      /* used to restart path */
       *lastSrchE = NULL;      /* used to restart match expr */
  int   fail      = 0;         /* TRUE if the match failed   */

  while (!fail && *pathPtr && *exclPtr) {
    switch (*exclPtr) {
      case '?':
        /*
          match any single character. this is easy
        */
        pathPtr++;
        exclPtr++;
        break;
      case '*':
      {
        int minLen = 0,
            len = 0,
            ii,
            fnd = 0;
        /*
          first, clear out the case of multiple '*'s
        */
        lastSrchE = exclPtr;
        while (*exclPtr == '*')
          exclPtr++;
        /*
          next, look for the minimum number of characters
        */
        while (*exclPtr == '?') {
          minLen++;
          exclPtr++;
        }
        if (!*exclPtr) {
          /*
            let's shortcut if we're looking at the end of the expression
            string
          */
          if (strlen(pathPtr) >= minLen)
            pathPtr += strlen(pathPtr);
          break;
        }
        /*
          ok, let's find out how long the substring must be
        */
        len=0;
        while (exclPtr[len]
               && (exclPtr[len] != '?')
               && (exclPtr[len] != '*'))
          len++;
        /*
          now:
            find the matching substring in pathPtr
        */
        fnd=0;
        while (!fnd && (strlen(pathPtr) >= len)) {
          for (ii=0; (ii < len) && (tolower(exclPtr[ii]) == tolower(pathPtr[ii])); ii++);
          if (ii != len)
            pathPtr++;
          else
            fnd = 1;
        }
        fail = !fnd;
        if (fnd) {
          lastSrchP = pathPtr+1;
          exclPtr += len;
          if (strlen(pathPtr) >= len+minLen)
            pathPtr += len + minLen;
        }
        break;
      }
      default:
        /*
          absolute match. this is easy also
        */
        if (tolower(*pathPtr) == tolower(*exclPtr)) {
          pathPtr++;
          exclPtr++;
        } else if (lastSrchP) {
          pathPtr = lastSrchP;
          exclPtr = lastSrchE;
          lastSrchP = lastSrchE = NULL;
        } else
          fail = 1;
    }
  }
  /*
    skip any '*'s at the end of the string
  */
  while (*exclPtr == '*')
    exclPtr++;
  /*
    if we're at the end of both strings, we've a match
  */
  return (!*pathPtr && !*exclPtr) ? TRUE : FALSE;
}

/*
  search [list] for [path]
  list = masks seperated by '\0' with "\0\0" at the end
*/
LOCAL int inMatchList(char *list, char *path)
{
  char *ptr;
  for (ptr = list; ptr && *ptr; ptr += strlen(ptr)+1)
    if (isMatch(ptr, path))
      return TRUE;
  return FALSE;
}

/*
  free all mem. used by the file list
*/
LOCAL void freeList(FILELIST *list)
{
  int ii;
  FILEINFO *inf;

  for (ii=0, inf = list->info; ii < list->ct; ii++, inf++)
    chkFree(inf->fullName);
  if (list->ct)
    chkFree(list->info);
  chkFree(list->basePath);
}

/*
  build a file list, adding the found files to [info]
  [include] = file lists to include
  [exclude] = file lists to exclude
  [path]    = full path to search
*/
LOCAL void buildList(FILELIST *info, char *include, char *exclude, char *path)
{
  char *fullPath = chkMalloc(128);
  char *namePos;
  int   done;
  struct ffblk ffblk;

  strcpy(fullPath, path);
  if (fullPath[strlen(fullPath)-1] != '\\')
    strcat(fullPath, "\\");
  namePos = fullPath + strlen(fullPath);
  strcat(fullPath, "*.*");
  for (done = findfirst(fullPath,
                        &ffblk,
                        FA_DIREC
                        | FA_ARCH
                        | FA_RDONLY
                        | FA_HIDDEN
                        | FA_SYSTEM);
       !done;
       done = findnext(&ffblk), fileCt++) {
    if ((fileCt & 0x0f) == 0) {
      cprintf("\rBuilding list (%s) processed (%d) accepted (%d)",
        info->basePath, fileCt, info->ct);
    }
    InterruptProgram();
    if (ffblk.ff_name[0] != '.') {
      strcpy(namePos, ffblk.ff_name);
      if (ffblk.ff_attrib & FA_DIREC)
        strcat(namePos, "\\");
      if (inMatchList(include, fullPath) && !inMatchList(exclude, fullPath)) {
        if (ffblk.ff_attrib & FA_DIREC) {
          if (flags & FLAG_RECURSE)
            buildList(info, include, exclude, fullPath);
        } else {
          FILEINFO *inf;
          if (!(info->ct & 0x0f)) {
            if (info->info)
              info->info = chkRealloc(info->info, (info->ct+0x10L) * sizeof(*info->info));
            else
              info->info = chkMalloc(0x10L * sizeof(*info->info));
          }
          inf = info->info + info->ct;
          memset(inf, 0, sizeof(*inf));
          info->ct++;
          inf->fullName = strdup(fullPath+strlen(info->basePath));
          inf->time = ffblk.ff_ftime;
          inf->date = ffblk.ff_fdate;
          inf->size = ffblk.ff_fsize;
          inf->attr = ffblk.ff_attrib;
          inf->action = 0;
        }
      }
    }
  }
  chkFree(fullPath);
  cprintf("\rBuilding list (%s) processed (%d) accepted (%d)",
    info->basePath, fileCt, info->ct);
}

LOCAL int  sortListCmp(const void *a, const void *b)
{
  FILEINFO *aInfo = a,
           *bInfo = b;
  return strcmp(aInfo->fullName, bInfo->fullName);
}

LOCAL void sortList(FILELIST *info)
{
  qsort(info->info, info->ct, sizeof(*info->info), sortListCmp);
}

LOCAL void subProcess(FILELIST *srcFileList, FILELIST *dstFileList)
{
  int ii;
  for (ii=0; ii < srcFileList->ct; ii++) {
    FILEINFO *inf = srcFileList->info+ii;
    FILEINFO *dst = bsearch(inf,
                            dstFileList->info,
                            dstFileList->ct,
                            sizeof(*dstFileList->info),
                            sortListCmp);
    if (dst) {
      if (flags & FLAG_UPDATE) {
        if ((inf->date > dst->date) ||
            ((inf->date == dst->date) && (inf->time > dst->time))) {
          inf->action = ACTION_UPDATE;
          srcFileList->updCt++;
          srcFileList->updBytes += inf->size;
        }
      }
    } else if (flags & FLAG_ADD) {
      inf->action = ACTION_ADD;
      srcFileList->addCt++;
      srcFileList->addBytes += inf->size;
    }
  }
}

LOCAL int CopyFile(char *srcName, char *srcPath, char *dstPath, int srcAttr)
{
  char dstName[128];
  char fullSrcName[128];
  int  inHandle  = 0;
  int  outHandle = 0;
  int  err       = 0;
  long toCopyFile,
       amtCopiedFile;

  strcpy(fullSrcName, srcPath);
  strcat(fullSrcName, srcName);
  inHandle = open(fullSrcName, O_RDONLY | O_BINARY);
  if (inHandle < 0)
    return _doserrno;

  toCopyFile    = filelength(inHandle);
  amtCopiedFile = 0;

  strcpy(dstName, dstPath);
  strcat(dstName, srcName);
  outHandle = open(dstName,
                O_RDWR | O_BINARY | O_CREAT | O_TRUNC,
                S_IREAD | S_IWRITE);
  if (outHandle < 0) {
    if (errno == EACCES) {
      if (_chmod(dstName, 1, 0) == -1)
        err = _doserrno;
    } else {
      /*
        try to create the path (don't worry about mkdir(..) fails)
      */
      char *ptr;
      for (ptr = strchr(dstName, '\\'); ptr; ptr = strchr(ptr+1, '\\')) {
        *ptr = 0;
        mkdir(dstName);
        *ptr = '\\';
      }
    }
    if (!err) {
      outHandle = open(dstName,
                    O_RDWR | O_BINARY | O_CREAT | O_TRUNC,
                    S_IREAD | S_IWRITE);
    }
  }
  if (outHandle < 0)
    err = _doserrno;
  else {
    char *buf = chkMalloc(16384);
    int   inLen, outLen;
    struct ftime ft;

    if (flags & FLAG_VERBOSE)
      UpdateVerboseScreen(amtCopiedFile, toCopyFile);

    do {
      InterruptProgram();
      inLen = read(inHandle, buf, 16384);
      if (inLen < 0)
        err=_doserrno;
      else {
        amtCopiedFile += inLen;
        InterruptProgram();
        outLen = write(outHandle, buf, inLen);
        if (outLen < 0)
          err = _doserrno;
        else
          outOfSpace = (outLen != inLen);
      }
      if (flags & FLAG_VERBOSE)
        UpdateVerboseScreen(amtCopiedFile, toCopyFile);
    } while (!err && (inLen == 16384));
    if (getftime(inHandle, &ft) == -1)
      err = _doserrno;
    else if (setftime(outHandle, &ft) == -1)
      err = _doserrno;
    if ((close(outHandle) == -1) && !err)
      err = _doserrno;
    else if (_chmod(dstName, 1, srcAttr) == -1)
      err = _doserrno;
    chkFree(buf);
  }
  close(inHandle);
  return (outOfSpace) ? -1 : err;
}

LOCAL int DeleteFile(char *name, char *path)
{
  char fullName[128];
  int  err = 0;

  strcpy(fullName, path);
  strcat(fullName, name);
  if (flags & FLAG_VERBOSE)
    UpdateVerboseScreen(0,0);
  if (unlink(fullName) == -1) {
    /*
      unlink failed, let's try reseting any attributes
    */
    _chmod(fullName, 1, 0);
    if (unlink(fullName) == -1)
      err = _doserrno;
  }
  if (!err && (flags & FLAG_DELDIR)) {
    /*
      delete the directory tree up to (file)
    */
    char *ptr = strrchr(fullName, '\\');
    int   ilen = strlen(path);
    while (ptr && ((ptr - fullName) > ilen)) {
      *ptr = 0;
      if (rmdir(fullName) == 0)
        ptr = strrchr(ptr, '\\');
      else if (errno == EACCES) {
        _chmod(fullName, 1, 0);
        if (rmdir(fullName) == -1)
          ptr = 0;
      } else
        ptr = 0;
    }
  }
  return err;
}

/*
  do the processing here
*/
LOCAL void ProcessList(void)
{
  /*
    first, let's process source
  */
  subProcess(&srcFiles, &dstFiles);
  if (flags & FLAG_BOTH)
    subProcess(&dstFiles, &srcFiles);
  else if (flags & FLAG_DELETE) {
    int ii;
    for (ii=0; ii < dstFiles.ct; ii++) {
      FILEINFO *inf = dstFiles.info+ii;
      FILEINFO *src = bsearch(inf,
                              srcFiles.info,
                              srcFiles.ct,
                              sizeof(*srcFiles.info),
                              sortListCmp);
      if (!src) {
        inf->action = ACTION_DELETE;
        dstFiles.delCt++;
        dstFiles.delBytes += inf->size;
      }
    }
  }
}

LOCAL void showHelp(char *errMessage)
{
  cprintf("SYNCDIR src [dest] [flags]\n"
         "  synchronise file/directory trees. if [dst] is missing, the current\n"
         "  directory is assumed\n"
         "flags:\n"
         "\t/a  add files found in [src] not in [dst]\n"
         "\t/d[d]  delete files in [dst] not in [src]\n"
         "\t/u  update files in [dst] which are older than like-named files in [src]\n"
         "\t/b  do action in both directions.\n"
         "\t/f[aud] do not prompt on add, update, delete (force mode)\n"
         "\t/r  recurse subirectories\n"
         "\t/v  verbose mode\n"
         "\t/n  do not perform any action.\n"
         "\t/l[filename] log actions to [filename] or SYNCDIR.LOG if none\n"
         "\t/i... include file list, seperated by ';' (default = \"*\")\n"
         "\t/x... exclude file list, seperated by ';' (default = none)\n"
         "\n");
  if (errMessage)
    cprintf("%s\n", errMessage);
  exit(2);
}

/*
  add / update / delete files

  i'm going
*/
LOCAL void ProcessFiles(FILELIST *lst, char *basePath, int isDst, int action)
{
  int ii;
  FILEINFO *inf;
  char *actionText[] = {"no action", "adding", "updating", "deleting"};
  int frcResponse = 0;

  outOfSpace = FALSE;
  for (ii=0, inf = lst->info;
       !finishAndAbort && (ii < lst->ct);
       ii++, inf++) {
    InterruptProgram();
    if ((inf->action == action) || (flags & FLAG_NOACTION)) {
      int OK = TRUE;
      char *namePtr = inf->fullName;
      char *fileText;

      if (scrollPos > 8) {
        char *tmp = scrollList[0];
        int   ii;
        for (ii=0; ii < 8; ii++)
          scrollList[ii] = scrollList[ii+1];
        scrollList[8] = tmp;
      }
      fileText = scrollList[min(scrollPos, 8)];
      if (strlen(namePtr) > 53) {
        while (strlen(namePtr) > 49)
          namePtr = strchr(namePtr+1, '\\');
      }
      sprintf(fileText,
        "[%s] {%s}%s%s",
        actionText[inf->action],
        (isDst) ? "dst" : "src",
        (strlen(inf->fullName) > 53) ? "..." : "",
        namePtr);
      if (flags & FLAG_VERBOSE)
        UpdateVerboseScreen(0,0);
      else
        cprintf("%s\r", fileText);
      /*
        prompt if the necessary action flags are not switched ``on''
        don't prompt if out of disk space (no action will occur anyway)
      */
      if (!(flags & FLAG_NOACTION)
          && ((!outOfSpace && (inf->action == ACTION_ADD) && !(flags & FLAG_FORCEADD))
          || (!outOfSpace && (inf->action == ACTION_UPDATE) && !(flags & FLAG_FORCEUPD))
          || ((inf->action == ACTION_DELETE) && !(flags & FLAG_FORCEDEL)))) {
        int response;
        if (frcResponse)
          response = frcResponse;
        else {
          if (flags & FLAG_VERBOSE)
            gotoxy(1, 14+min(8, scrollPos));
          cprintf("%s Exec?", fileText);
          do {
            response = getch();
          } while (!strchr("yYnN", response));
          if (response == 'Y')
            response = frcResponse = 'y';
          else if (response == 'N')
            response = frcResponse = 'n';
        }
        OK = (response != 'n');
      }
      if (flags & FLAG_NOACTION) {
        if (logFile && (inf->action || (flags & FLAG_VERBOSE)))
          fprintf(logFile, "%s\n", fileText);
      } else if (!OK) {
        strcat(fileText, " SKIPPED");
        if (logFile)
          fprintf(logFile, "%s\n", fileText);
      } else {
        if ((inf->action == ACTION_ADD) || (inf->action == ACTION_UPDATE)) {
          if (outOfSpace || (CopyFile(inf->fullName, lst->basePath, basePath, inf->attr) != 0)) {
            strcat(fileText, " ERROR");
            if (logFile)
              fprintf(logFile, "%s\n...%s",
                 fileText,
                 (outOfSpace) ? "Out of disk space\n" : _strerror(""));
          } else {
            strcat(fileText, " OK   ");
            if (logFile)
              fprintf(logFile, "%s\n", fileText);
          }
        } else if (inf->action == ACTION_DELETE) {
          if (DeleteFile(inf->fullName, lst->basePath)) {
            strcat(fileText, " ERROR");
            if (logFile)
              fprintf(logFile, "%s\n...%s",
                fileText, _strerror(""));
          }
        }
      }
      if (!(flags & FLAG_VERBOSE))
        cprintf("\r%s\r\n", fileText);
      if (scrollPos < 9)
        scrollPos++;
      switch (inf->action) {
        case ACTION_ADD:
          addFile--;
          addBytes -= inf->size;
          break;
        case ACTION_DELETE:
          delFile--;
          delBytes -= inf->size;
          break;
        case ACTION_UPDATE:
          updFile--;
          updBytes -= inf->size;
          break;
      }
      if (flags & FLAG_VERBOSE)
        UpdateVerboseScreen(0,0);
    }
  }
}

LOCAL char *GrabPath(char **pptr)
{
  char *ptr = *pptr;
  char *ePtr = ptr;
  char *result = NULL;
  int   len;

  while (isspace(*ptr))
    ptr++;

  for (ePtr = ptr; *ePtr && (*ePtr != SWITCHCHAR) && !isspace(*ePtr); ePtr++);
  len = ePtr-ptr;
  result = chkMalloc(len+4);
  strncpy(result, ptr, len);
  if (result[len-1] == '\\')
    len--;
  else if (result[len-1] == ':')
    result[len++]='.';
  result[len]=0;

  while (isspace(*ePtr))
    ePtr++;
  *pptr = ePtr;

  return result;
}

GLOBAL int main(int argc, char **argv)
{
  char *ptr;
  char logFileName[256];
  int  len;

  /*
    create full command line in paramStr
  */
  char *paramStr = chkMalloc(16384);

  logFileName[0] = 0;
  strcpy(paramStr, argv[1]);
  {
    int ii;
    for (ii=2; ii < argc; ii++) {
      strcat(paramStr, " ");
      strcat(paramStr, argv[ii]);
    }
  }
  {
    char *tmp = getenv("SYNCDIR");
    if (tmp) {
      strcat(paramStr, " ");
      strcat(paramStr, tmp);
    }
  }
  cos(1.0);  /* force math lib to be linked */

  ptr = paramStr;
  while (isspace(*ptr))
    ptr++;
  if (*ptr == SWITCHCHAR)
    showHelp("no source path found");
  srcFiles.basePath = GrabPath(&ptr);
  if (*ptr != SWITCHCHAR)
    dstFiles.basePath = GrabPath(&ptr);
  else {
    dstFiles.basePath = chkMalloc(128);
    getcwd(dstFiles.basePath, 128);
  }

  while (*ptr == SWITCHCHAR) {
    ptr++;
    switch (*(ptr++)) {
      case 'a': flags |= FLAG_ADD;      break;
      case 'b': flags |= FLAG_BOTH;     break;
      case 'f':
      {
        int ct = 0;
        for ( ; ; ) {
          if (*ptr == 'a') {
            flags |= FLAG_FORCEADD;
            ptr++;
            ct++;
          } else if (*ptr == 'd') {
            flags |= FLAG_FORCEDEL;
            ptr++;
            ct++;
          } else if (*ptr == 'u') {
            flags |= FLAG_FORCEUPD;
            ptr++;
            ct++;
          } else
            break;
        }
        if (!ct)
          flags |= (FLAG_FORCEADD | FLAG_FORCEUPD | FLAG_FORCEDEL);
        break;
      }
      case 'r': flags |= FLAG_RECURSE;  break;
      case 'v': flags |= FLAG_VERBOSE;  break;
      case 'n': flags |= FLAG_NOACTION; break;
      case 'u': flags |= FLAG_UPDATE;   break;
      case 'd':
        if (*ptr == 'd') {
          ptr++;
          flags |= (FLAG_DELETE | FLAG_DELDIR);
        } else
          flags |= FLAG_DELETE;
        break;
      case 'i':
        if (includeFiles[0])
          strcat(includeFiles, ";");
        for (len=0; ptr[len] && !isspace(ptr[len]) && (ptr[len] != SWITCHCHAR); len++);
        strncat(includeFiles, ptr, len);
        ptr+=len;
        break;
      case 'x':
        if (excludeFiles[0])
          strcat(excludeFiles, ";");
        for (len=0; ptr[len] && !isspace(ptr[len]) && (ptr[len] != SWITCHCHAR); len++);
        strncat(excludeFiles, ptr, len);
        ptr += len;
        break;
      case 'l':
        for (len=0; ptr[len] && !isspace(ptr[len]) && (ptr[len] != SWITCHCHAR); len++);
        if (len) {
          strncpy(logFileName, ptr, len);
          logFileName[len]=0;
        } else
          strcpy(logFileName, "syncdir.log");
        ptr += len;
        break;
      default:
        {
          char buf[64];
          sprintf(buf, "unknown flag {%s}", *(ptr-1));
          showHelp(buf);
        }
    }
    while (isspace(*ptr))
      ptr++;
  }
  if (*ptr) {
    char buf[64];
    sprintf(buf, "extra characters in flag {%s}", ptr);
    showHelp(buf);
  }
  if ((flags & FLAG_BOTH) &&
      (flags & FLAG_DELETE))
    showHelp("cannot use both /b and /d or /dd");

  if (!includeFiles[0])
    strcpy(includeFiles, "*");

  chkFree(paramStr);
  if (logFileName[0]) {
    logFile = fopen(logFileName, "wt");
    if (!logFile) {
      char buf[64];
      sprintf(buf, "Cannot create [%s]", logFileName);
      perror(buf);
      exit(4);
    }
  }
  /*
    now, create a list of source files
  */
  cprintf("Exclude: %s\n"
         "Include: %s\n"
         "\n",
         excludeFiles,
         includeFiles);
  if (logFile)
    fprintf(logFile,
           "Exclude: %s\n"
           "Include: %s\n"
           "\n",
           excludeFiles,
           includeFiles);

  for (ptr = strchr(includeFiles, ';'); ptr; ptr = strchr(ptr+1, ';'))
    *ptr=0;
  for (ptr = strchr(excludeFiles, ';'); ptr; ptr = strchr(ptr+1, ';'))
    *ptr=0;
  fileCt = 0;
  buildList(&srcFiles, includeFiles, excludeFiles, srcFiles.basePath);
  if (!finishAndAbort) {
    sortList(&srcFiles);
    cprintf("\n\r");
    fileCt = 0;
    buildList(&dstFiles, includeFiles, excludeFiles, dstFiles.basePath);
    if (!finishAndAbort) {
      sortList(&dstFiles);
      cprintf("\n\r");
      ProcessList();
      startTime = clock();

      addFile = srcFiles.addCt + dstFiles.addCt;
      updFile = srcFiles.updCt + dstFiles.updCt;
      delFile = dstFiles.delCt;

      addBytes = srcFiles.addBytes + dstFiles.addBytes;
      updBytes = srcFiles.updBytes + dstFiles.updBytes;
      delBytes = dstFiles.delBytes;

      toCopy = addBytes + updBytes;

      {
        int ii;
        for (ii=0; ii < 10; ii++)
          scrollList[ii] = chkMalloc(80);
      }
      if (flags & FLAG_VERBOSE)
        InitVerboseScreen();

      {
        int ii;
        for (ii = ACTION_DELETE; !finishAndAbort && (ii >= ACTION_ADD); ii--) {
          ProcessFiles(&srcFiles, dstFiles.basePath, FALSE, ii);
          if (!finishAndAbort)
            ProcessFiles(&dstFiles, srcFiles.basePath, TRUE, ii);
        }
      }
    }
  }
  freeList(&srcFiles);
  freeList(&dstFiles);
  if (logFile) {
    if (finishAndAbort)
      fprintf(logFile, "<<user interrupt --- last action completed>>\n");
    fclose(logFile);
  }
  {
    int ii;
    for (ii=0; ii < 10; ii++)
      chkFree(scrollList[ii]);
  }

  return 0;
}
