/*
 *  MsgPump2 uses a secondary message loop to keep the messages flowing
 *  while the window is filled with ellipses.
 */

#define IDD_START       100
#define IDD_STOP        101
#define BUTTONWIDTH      72
#define BUTTONHEIGHT     24
#define NUMELLIPSES    2048

#include <windows.h>
#include <stdlib.h>
#include <time.h>

long FAR PASCAL WndProc (HWND, WORD, WORD, LONG);
void DrawEllipse (HWND);
void CheckMessageQueue (void);

/*
 *  WinMain starts the program.
 */

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    static char szAppName[] = "MsgPump2";
    WNDCLASS wc;
    HWND hwnd;
    MSG msg;

    if (!hPrevInstance) {
        wc.style = CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc = (WNDPROC) WndProc;
        wc.cbClsExtra = 0;
        wc.cbWndExtra = 0;
        wc.hInstance = hInstance;
        wc.hIcon = LoadIcon (NULL, IDI_APPLICATION);
        wc.hCursor = LoadCursor (NULL, IDC_ARROW);
        wc.hbrBackground = COLOR_WINDOW + 1;
        wc.lpszMenuName = NULL;
        wc.lpszClassName = szAppName;

        RegisterClass (&wc);
    }

    hwnd = CreateWindow (szAppName, szAppName, WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
        NULL, NULL, hInstance, NULL);

    ShowWindow (hwnd, nCmdShow);
    UpdateWindow (hwnd);

    while (GetMessage (&msg, NULL, 0, 0)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }

    return msg.wParam;
}

/*
 *  WndProc processes messages to the main window.
 */

long FAR PASCAL WndProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    HDC hdc;
    PAINTSTRUCT ps;
    static BOOL bContinue;
    RECT rect;
    UINT i;
        
    switch (message) {

    case WM_CREATE:
        //
        // Create the Start and Stop buttons and seed the random number
        // generator.
        //
        CreateWindow ("button", "Start",
            WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON,
            0, 0, BUTTONWIDTH, BUTTONHEIGHT, hwnd, IDD_START,
            (HINSTANCE) ((LPCREATESTRUCT) lParam)->hInstance, NULL);

        CreateWindow ("button", "Stop",
            WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON | WS_DISABLED,
            BUTTONWIDTH, 0, BUTTONWIDTH, BUTTONHEIGHT, hwnd, IDD_STOP,
            (HINSTANCE) ((LPCREATESTRUCT) lParam)->hInstance, NULL);

        srand ((unsigned) time (NULL));
        return 0;       

    case WM_PAINT:
        //
        // Draw the background behind the push buttons.
        //  
        hdc = BeginPaint (hwnd, &ps);
        GetClientRect (hwnd, &rect);
        rect.bottom = BUTTONHEIGHT;
        FillRect (hdc, &rect, (HBRUSH) GetStockObject (LTGRAY_BRUSH));
        MoveTo (hdc, 0, BUTTONHEIGHT);
        LineTo (hdc, rect.right, BUTTONHEIGHT);
        EndPaint (hwnd, &ps);
        return 0;

    case WM_COMMAND:
        //
        // Process push button messages.
        //
        switch (wParam)  {
        
        case IDD_START:
            //
            // Start drawing.
            //
            EnableWindow (GetDlgItem (hwnd, IDD_START), FALSE);
            EnableWindow (GetDlgItem (hwnd, IDD_STOP), TRUE);

            i = 0;
            bContinue = TRUE;

            while ((i++ < NUMELLIPSES) && bContinue) {
                DrawEllipse (hwnd);
                CheckMessageQueue ();
            }

            EnableWindow (GetDlgItem (hwnd, IDD_START), TRUE);
            EnableWindow (GetDlgItem (hwnd, IDD_STOP), FALSE);
            return 0;       
        
        case IDD_STOP:
            //
            // Stop drawing.
            //
            bContinue = FALSE;
            return 0;
        }
        break;

    case WM_DESTROY:
        //
        // Terminate the program.
        //
        PostQuitMessage (0);
        return 0;
    }
    return DefWindowProc (hwnd, message, wParam, lParam);
}

/*
 *  DrawEllipse draws an ellipse of an arbitrary size and color in the
 *  window's client area.
 *
 *  Input parameters:
 *    hwnd = Window handle
 *
 *  Returns:
 *    Nothing
 */
 
void DrawEllipse (HWND hwnd)
{
    RECT rect;
    HBRUSH hBrush;
    int x1, y1, x2, y2;
    HDC hdc;
            
    GetClientRect (hwnd, &rect);
    rect.bottom -= BUTTONHEIGHT;
    
    x1 = rand () % rect.right;
    x2 = rand () % rect.right;
    y1 = (rand () % rect.bottom) + BUTTONHEIGHT;
    y2 = (rand () % rect.bottom) + BUTTONHEIGHT;
    
    hBrush = CreateSolidBrush (RGB (rand () & 255, rand () & 255,
        rand () & 255));    
    hdc = GetDC (hwnd);
    SelectObject (hdc, hBrush);

    Ellipse (hdc, min (x1, x2), min (y1, y2), max (x1, x2), max (y1, y2));

    ReleaseDC (hwnd, hdc);
    DeleteObject (hBrush);
}

/*
 *  CheckMessageQueue retrieves and dispatches any messages waiting in
 *  the application's message queue.
 *
 *  Input parameters:
 *    None
 *
 *  Returns:
 *    Nothing
 */
 
void CheckMessageQueue (void)
{
    MSG msg;

    while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }
}
