/*  VIRLIST V1.2.1
 *
 *  Read McAfee's VIRSCAN VIRLIST.TXT file and display viruses
 *  with selected characteristics; (e.g., all self-encrypting stealth viruses
 *  with length >= 1024 bytes).
 *
 *    Program:  VIRLIST Version 1.2.1
 *    Platform:	MS-DOS/Borland C V3.1, UNIX/gcc
 *    Author:	Jason Mathews
 *    Date:     12-May-94
 *
 * Copyright (C) 1992-94 by Jason Mathews.  Permission is granted to any
 * individual or institution to use, copy or redistribute this software so long
 * as it is not sold for profit, provided this copyright notice is retained.
 *
 *  History:
 *   2/24/92 - Developed original program.
 *   6/12/92 - Added negative logic to filter out characteristics,
 *	       added logic for checking virus length (LT, EQ, GT).
 *   8/15/93 - Added LE, GE size flags, fixed size checking code
 *             for variable string position within size field,
 *             optimized damage field checking,
 *	       added fuzzy string search for single typos in virlist text.
 *   5/12/94 - Added Companion and Unknown keywords.
 */

#include "virlist.h"

int main (int argc, char **argv)
{
	int matches = 0;
	int misses  = 0;
	FILE *fp = fopen( VIRLIST_FILE, "r" );

	puts("VIRLIST V1.2.1 Copyright 1992-94 by Jason Mathews\n");

	if (fp==0)
	{
		fprintf(stderr, "Cannot file virus list '%s'\n", VIRLIST_FILE);
		fprintf(stderr, "Please copy this file into the appropriate directory.\n");
		return 1;
	}

	CheckArgs(argc, argv);		/* Check arguments */

	SkipHeader(fp);			/* Skip the virlist.txt header */

	while (fgets(buffer, 1024, fp))
	{
		/*
		 * stop when the two blank lines at the end are processed.
		 */
		if (*buffer == '\n' && ++misses == 2) break;
		else if (GetRecord(buffer) > 0)
		{
			PrintRecord();
			matches++;
			misses = 0;
		}
	}
	fclose(fp);

	printf("\nTotal of %d match%s found\n",
	       matches, (matches!=1) ? "es" : "");

	if (showLegend)
	  Legend();			/* Show code legend and exit */
	return 0;
}  /** main **/


/* CheckArgs:
 *  Check program flags & options
 */
void CheckArgs (int argc, char **argv)
{
  int i, offset;
  char opt;

  if (argc == 1) Help( 0, NULL );

  printf("Program options:\n");
  for (i=1; i < argc; i++)
    {
      if (*argv[i] == '-' || *argv[i] == '+')
        {
	  switch (argv[i][1]) {
	  case 's':
#ifdef __TURBOC__
	    strlwr(argv[i]); /* convert to lower case */
#endif
	    switch (argv[i][2]) {
	    case 'e':
	      size_code = EQUAL;
	      break;
	    case 'l':
	      size_code = (argv[i][3] == 'e' ? LESS_EQUAL : LESS_THAN);
	      break;
	    case 'g':
	      size_code = (argv[i][3] == 'e' ? GREATER_EQUAL : GREATER_THAN);
	      break;
	    case 'c':
	      size_code = COMPANION;
	      puts("\tmatch all companion viruses");
	      break;	      
	    case 'o':
	      size_code = OVERWRITE;
	      puts("\tmatch all overwriting viruses");
	      break;
	    case 'v':
	      size_code = VARIES;
	      puts("\tmatch all variable length viruses");
	      break;
	    case 's':
	      size_code = SPAWN;
	      puts("\tmatch all spawning viruses");
	      break;
	    default:
	      size_code = INVALID;
	      Help( ERR_SIZE, argv[i] );
	    } /* switch on size code */

	    if (size_code > INVALID) /* length associated w/parameter */
	      {
		char *s;
		if (strlen(argv[i]) > 4) s = argv[i]+4;
		else if (i < argc-1) s = argv[++i]; /* try next argument? */
		else 
		{
		  Help( ERR_SIZE, argv[i] ); /* bad usage - abor\t */
		}
		size_value = atol(s); 
		if (size_value < 0) Help( ERR_SIZE, argv[i] ); /* bad value */
		printf("\t+s - Match virus size %s %ld bytes\n",
		       size_strings[size_code-1], size_value);

		if (size_code == GREATER_EQUAL)
		  {
		    /* m >= n same as m > n-1 */
		    --size_value;
		    size_code = GREATER_THAN;
		  }
		else if (size_code == LESS_EQUAL)
		  {
		    /* m <= n same as m < n+1 */
		    ++size_value;
		    size_code = LESS_THAN;
		  }
	      }
	    break;    /* end case '-s' argument */

	  default:
	    /* parse argument string */
	    for (offset=1; argv[i][offset] != 0; offset++)
	      switch (tolower(argv[i][offset])) /* check letter codes */
	      {
	        case 'b': SetField (D_BOOT, 1, *argv[i]);
		          break;
		case 'p': SetField (D_PROGRAM, 2, *argv[i]);
			  break;
		case 'd': SetField (D_DATA, 3, *argv[i]);
			  break;
		case 'f': SetField (D_FORMAT, 4, *argv[i]);
			  break;
		case 'l': SetField (D_LINKS, 5, *argv[i]);
			  break;
		case 'o': SetField (D_OPS, 6, *argv[i]);
			  break;

		case 'a':
			SetFlag (10, 16, *argv[i]);
			break;

		case 'h':
		case '?':
			Help(0, NULL);
			break;

		default:
			opt = argv[i][offset];
			if (opt>='1' && opt<='9') /* check for options 1-9 */
			{
				SetFlag (opt-'0', opt-'0'+6, *argv[i]);
			}
			else Help( ERR_FLAG, argv[i]+offset );
			break;
		} /* switch */
	  } /* switch */
	}/* if */
      else if (!strcmp(argv[i], "nl") || !strncmp(argv[i], "nol", 3))
	showLegend = FALSE;
      else
	Help( ERR_OPT, argv[i] ); /* unknown option - abort */
    } /* for */

  puts(messages[0]);  /* show a dashed line */
}  /** CheckArgs **/


/*
 * CheckCode:
 *   Check if code is set for entry and
 *   print its corresponding letter or a blank.
 */
void CheckCode (int code, char letter)
{
    /* first check for potential characteristics with (?) code */
    if (virus.codes[code-1] == '?') putchar('?');
    else putchar( (virus.codes[code-1]) ? letter : '.');

    putchar(' '); /* separate codes with blank */
}  /** CheckCode **/


/* GetRecord:
 *   Get fields from virlist entry and fill in record
 *
 * Returns:  positive no. if match found,
 *           0 if no match found,
 *          -1 if entry is too small.
 */
int GetRecord (char *buf)
{
    int i, length = strlen(buf);
    char tmp[20];	/* temp buffer */
    int match = 0;   	/* number of matches */
    register char* pbuf;

    if (length < 58) return(-1); /* entry too short - no codes to check */

    /* strip off virus name and disinfector */

    Copy(name, 26);
    buf += 26;
    Copy(disinfector, 9);
    buf += 13;

    /* Parse each virus code and check value for 'x', '?', or blank */

    for (i=0; i < 10; i++)
    {
	if (*buf == '?')  /* take a '?' as maybe */
	{
		if (flags[i] != NOT_USED)
		{
			match++;
			virus.codes[i] = '?';
		}
	}
	else if (*buf == 'x')
        {
		virus.codes[i] = 1;
		if (flags[i]==TRUE) match++;
        else if (flags[i] == FALSE) return 0; /* no match */
	}
	else  /* must be a blank */
	{
		virus.codes[i] = 0;
		if (flags[i]==FALSE) match++;
		else if (flags[i] == TRUE) return 0; /* no match */
	}
	buf += 2;
    } /* for */

    /* check virus size */
    pbuf = buf;
    while (*pbuf==' ' || *pbuf=='\t') pbuf++;

    /* check for actual number in bytes */
    if (*pbuf >= '0' && *pbuf <= '9')
    {
	int len = 0;
	do {
	  tmp[len++] = *pbuf++;
	} while (len < 10 && *pbuf >= '0' && *pbuf <= '9');
	tmp[len] = '\0'; 
	virus.size = atol(tmp);
	virus.sizecode = SIZE_COMPUTED;
    }

/* use fuzzy text search for size codes to allow for typos,
 * which is slightly slower but more thorough
 * than just using strnicmp().
 */

    else if (strmatch(pbuf, "overwrite", 9))
	virus.sizecode = OVERWRITE;
    else if (strmatch(pbuf, "varies", 6))
	virus.sizecode = VARIES;
    else if (strmatch(pbuf, "unknown", 7))
        virus.sizecode = UNKNOWN;
    else if (strmatch(pbuf, "spawn", 5))
        virus.sizecode = SPAWN;
    else if (strmatch(pbuf, "companion", 9))
        virus.sizecode = COMPANION;
    /* otherwise handle other codes: N/A, See Note, etc. */
    else virus.sizecode = OTHER;

    switch (size_code) {
      case OVERWRITE:
        if (virus.sizecode == OVERWRITE) match++;
	else return(0); /* no match */
	break;
      case VARIES:
	if (virus.sizecode == VARIES) match++;
	else return(0); /* no match */
	break;
      case SPAWN:
	if (virus.sizecode == SPAWN) match++;
	else return(0);  /* no match */
	break;
      case COMPANION:
        if (virus.sizecode == COMPANION) match++;
	else return(0); /* no match */
	break;
      case EQUAL:
	if (virus.sizecode==SIZE_COMPUTED && virus.size==size_value)
	  match++;
	else return(0);
	break;
      case GREATER_THAN:
	if (virus.sizecode == SIZE_COMPUTED && virus.size > size_value)
	  match++;
	else return(0);
	break;
      case LESS_THAN:
	if (virus.sizecode == SIZE_COMPUTED && virus.size < size_value)
	  match++;
	else return(0);
    }

    while (*pbuf && *pbuf > ' ') pbuf++;   /* skip over size field */

    virus.damage = 0;	/* reset damage flags for virlist entry */

    /* Check damage fields */
    while (*pbuf)
    {
        switch (*pbuf++)
        {
	      case 'B': Damage(D_BOOT);
	      case 'D': Damage(D_DATA);
	      case 'F': Damage(D_FORMAT);
	      case 'L': Damage(D_LINKS);
	      case 'O': Damage(D_OPS);
	      case 'P': Damage(D_PROGRAM);
        } /* switch */
    }

    if (enableMask)
	if (enableMask & virus.damage) match++;
	else return 0;

    if (disableMask)
	if (disableMask & virus.damage) return 0;
	else match++;

    return match;
}  /** GetRecord **/


/* Help:
 *   Show program options
 */
void Help(int errcode, char* msg)
{
    int i;

    switch (errcode)
      {
      case ERR_SIZE:
	fprintf(stderr, "\tInvalid virus size usage: '%s'\n\n", msg);
	break;
      case ERR_FLAG:
	fprintf(stderr, "\tInvalid option/flag: %c\n\n", *msg);
	break;
      case ERR_OPT:
	fprintf(stderr, "\tInvalid option: '%s'\n\n", msg);
      }

    puts("Usage:\n\tvirlist [+|- BPDFLO123456789A] [-s<Size_code> [Size]] [nl]\n");
    puts("          '+' for the characterisic to be true");
    puts("          '-' for the characterisic to be false\n");
    puts("Options:");
    for (i=1; i < MSG_NUM; i++)
    {
	printf("\t%s\n", messages[i]);
#ifdef __TURBOC__
	if (i==15)
	{
	  fprintf(stderr, "press any key to continue");
	  getch();
	  fputc('\n', stderr);
	}
#endif
    }

    puts("\nSize codes:\n\t-s<code><size>, where code = logical operator { lt, le, eq, ge, gt }");
    puts("\t\t\t      size = comparison length");
    /* comparison operators followed by a number"); */

    puts("\t-so = show only overwriting viruses");
    puts("\t-sv = show only varied length viruses");
    puts("\t-ss = show only spawning viruses");
    puts("\nParameters:\n\tnl = do not display an option legend\n");
    puts("\n\nFor example: type 'virlist -slt1000 +2' to display viruses");
    puts("\t     that are less than 1000 bytes and use self-encryption");
    exit(1);
}  /** Help **/


/* Legend:
 *  Show code legend
 */
void Legend()
{
    int i;
    putchar('\n');
    puts(messages[0]);
    puts("Legend:");
    for (i=1; i < MSG_NUM; i++)
    {
        /* check for start of number codes to insert a new line */
        if (i==7) putchar('\n');
        printf("\t%s\n", messages[i]);
    }
}  /** Legend **/


/* PrintRecord:
 *  Print virlist record
 */
void PrintRecord()
{
    int i;
    printf("%s  ", virus.name);

	/*	Check if code is set and print letter if set or a blank if not
	 *	Note that the 10th code uses a 'P' instead of a number.
	 */
    for (i=1; i <= 10; i++)
    CheckCode (i, (i<10 ? i+'0' : 'A') );

    switch (virus.sizecode)
      {
      case OVERWRITE:
	printf("Overwrites");
	break;
      case VARIES:
        printf("  varies  ");
        break;
      case COMPANION:
        printf("Companion ");
	break;
      case UNKNOWN:
	printf(" Unknown  ");
	break;
      case OTHER:
        printf("    N/A   ");
        break;
     case SPAWN:
	printf(" Spawning ");
	break;
     case SIZE_COMPUTED:
	printf("%7ld   ", virus.size);
        break;
     default:
        printf("          ");
    }

    if (virus.damage & D_BOOT) printf(" B");
    if (virus.damage & D_DATA) printf(" D");
    if (virus.damage & D_FORMAT) printf(" F");
    if (virus.damage & D_LINKS) printf(" L");
    if (virus.damage & D_OPS) printf(" O");
    if (virus.damage & D_PROGRAM) printf(" P");
    putchar('\n');
}  /** PrintRecord **/

/* SetField:
 *  Set damage fields on/off and display corresponding action
 */
void SetField (WORD flag, short msg_num, char action)
{
  if (action=='+') enableMask |= flag;
  else disableMask |= flag;
  printf("\t%c%s - %s\n", action, messages[msg_num],
	 (action=='+' ? "true" : "false"));
}

/* SetFlag:
 *  Set option flag on/off and display corresponding action
 */
void SetFlag (int field, int msg_num, char action)
{
    /*  set flag code to TRUE for '+' action code
     *  and FALSE for '-' action code.
	 */
    flags[field-1] = (action == '+');
    printf("\t%c%s - %s\n", action, messages[msg_num],
	   (action=='+' ? "true" : "false"));
}  /** SetFlag **/

/* SkipHeader:
 *  Read past 'virlist.txt' header, which ends with a line of all dashes (-)
 *  The file location should now be at the beginning of the virlist entries.
 */
void SkipHeader (FILE *fp)
{
	while (fgets(buffer, 1024, fp))
	{
		if (!strncmp(buffer, "------", 6)) return;
	}
}  /** SkipHeader **/


#ifndef QUICK_SEARCH

/* strmatch:
 *  Fuzzy string match allowing one typo in case-insensitive string comparisons
 *  (assumes s2 is already lowercase)
 */
int strmatch( char *s1, char *s2, int len )
{
  int misses = 0;

  while (*s1 && *s2 && len > 0)
  {
    if (*s1 != *s2)
    {
	/* allow one typo in string match */
	if (tolower(*s1) != *s2 && misses++)
	 break;
    }
    ++s1;
    ++s2;
    --len;
  } /* while */

  return (!len);
}
#endif

