/*===========================================================================
SOLAR slrreply :: Module ..\slrreply\extract.c
Original author : Kevin Houle <kjhoule@iowegia.des-moines.ia.us>

This source code has been released into the public domain.
===========================================================================*/

/* Header Files */
#include <string.h>
#include <stdio.h>
#include <dir.h>
#include <stdlib.h>
#include <time.h>
#include "extract.h"

/*
 * Function: extract_msg(FILE *msg_file, char msg_type)
 * Purpose : Extract a single message from a MSG file into a
 *           temporary file for processing. Most of the header
 *           processing is done here.
 * Return  : -2L = fatal error, sets _slrerr
 *           -1L = non-fatal error, sets _slrerr
 *            0L = end of file
 *            msg size in bytesL on success
*/

long extract_msg(FILE *msg_file)
{
  long   endpos       = 0L;       /* End of article in bytes from zero    */
  long   curpos       = 0L;       /* Current position in article in bytes */
  long   msg_size     = 0L;       /* Size of message processed in bytes   */
  char   bigendian[4];            /* Stores msg length in 4-byte value    */
  FILE   *work_file   = NULL;     /* Final message to feed mail/news pgm. */
  HEADER *header      = NULL;     /* Base of msg header linked list       */
  HEADER *current     = NULL;     /* Current node in msg header list      */
  HEADER *scanlist    = NULL;     /* Used to scan the header list         */

  /* Test for end of the file. This doesn't always
     catch the end for some reason. */

  if (feof(msg_file))
  {
    msg_size = 0L;
    goto ExitFunct;
  }

  /* Initialize endpos relative to the start
     of the message in the file */

  endpos = ftell(msg_file);

  /* Get binary message length. Check for end of file. */

  bigendian[0] = getc(msg_file);
  if (feof(msg_file))
  {
    msg_size = 0L;
    goto ExitFunct;
  }
  bigendian[1] = getc(msg_file);
  if (feof(msg_file))
  {
    msg_size = 0L;
    goto ExitFunct;
  }
  bigendian[2] = getc(msg_file);
  if (feof(msg_file))
  {
    msg_size = 0L;
    goto ExitFunct;
  }
  bigendian[3] = getc(msg_file);
  if (feof(msg_file))
  {
    msg_size = 0L;
    goto ExitFunct;
  }

  /* Find the message size to expect in bytes */

 if ((msg_size = binlen(bigendian)) < 0)
 {
   strcpy(_slrerr,"extract_msg(): cannot determine binary message length");
   goto ExitFunct;
 }

 /* Calculate the ending position of the message in the file.
    We really should test against the total length of the
    message file, not doing so puts a lot of faith in the
    reader's length encoding. */

 endpos += msg_size + 4L; /* Allow 4 bytes for size encoding field */

 /* Check the message's size in bytes to make
    sure it is within configured limits. */

 switch (msg_type) {

   /* Binary format news message */

   case 'B'  : if (msg_size > _maxnewssize)
               {
                 /* Report message as being too long. */

                 sprintf(_slrerr,"extract_msg(): maximum size of %lu bytes exceeded for news",msg_size);

                 /* Advance to end of message in file */

                 fseek(msg_file,endpos,SEEK_SET); /* Advance message */

                 /* Set a message size error flag and exit. */

                 msg_size = -1L;
                 goto ExitFunct;
               }
               break;

   /* Binary format mail message */

   case 'b'  : if (msg_size > _maxmailsize)
               {
                 /* Report message as being too long. */

                 sprintf(_slrerr,"extract_msg(): maximum size of %lu bytes exceeded for mail",msg_size);

                 /* Advance to end of message in file */
                 fseek(msg_file,endpos,SEEK_SET); /* Advance message */

                 /* Set a message size error flag and exit. */

                 msg_size = -1L;
                 goto ExitFunct;
               }
               break;
 }

  /* Read through the message header and store in memory. Since
     RFC format message headers are LF terminated, we will use
     fgets() here even though message type is binary. */

  while (fgets(msgbuf,BUFSIZE,msg_file) != NULL)
  {
    /* Set current byte position in message file */

    curpos = ftell(msg_file);

    /* Check to see if we are at the end of the header. If
       so, break from read loop. This assumes RFC format
       messaging, where a single linefeed separates header
       from message. */

    if (strlen(msgbuf) < 3) break;

    if ((msg_type == 'B') && (strnicmp(msgbuf,"Newsgroups:",11) == 0))
    {
      /* If this is a type 'news' message and we have
         found the 'Newsgroups:' header, and gotta scan the
         forum files to check for moderated groups. */

      switch(scan_forums(msgbuf)) {

        /* A fatal error occurred */
        case -1   : msg_size = -2L; /* Fatal error */
                    goto ExitFunct;

        /* Article is OK to post */
        case  0   : break;

        /* Group is marked /nosolar */
        case  1   : sprintf(_slrerr,"extract_msg(): group marked for no posting");
                    fseek(msg_file,endpos,SEEK_SET); /* Advance message */
                    msg_size = -1L;                  /* Non-fatal error */
                    goto ExitFunct;

        /* Group is moderated, re-write as e-mail. */
        case  2   : msg_type = 'b';
                    RPrintf("To moderator: %s\n",mod_address);
                    sprintf(msgbuf,"To: %s\n",mod_address);
                    break;
      }

      /* Let's also count number of newsgroups to filter spams. */

      if (spam_count() > spam_max)
      {
        RPrintf("No SPAM allowed! Article being dropped to the floor.\n");
        sprintf(_slrerr,"spam alert!, %lu newsgroups",spam_count());
        msg_size = -1L; /* Non-fatal error */
        goto ExitFunct;
      }
    }

    /* Check header to see if allowed by configuration */

    if (allow_header(msgbuf) == 0)
    {
      if (!header)
      {
        /* First line of header. Create header object. */

        header = current = malloc(sizeof(HEADER));
        if (!current)
        {
          strcpy(_slrerr,"extract_msg(): insufficient memory to store header.");
          msg_size = -2L; /* Fatal error */
          goto ExitFunct;
        }
        header->next = NULL;
        strcpy(header->info,msgbuf);
        header->used = NO;
      }
      else
      {
        /* Trace thru existing headers. If this one isn't already listed */
        /* add the header to the end of the list.                        */

        scanlist = header;
        while (scanlist)
        {
          if (strcmp(msgbuf,scanlist->info) == 0) break;
          scanlist = scanlist->next;
        }
        if (!scanlist)
        {
          /* Create a new node in the list */
          current->next = malloc(sizeof(HEADER));
          if (!current->next)
          {
            strcpy(_slrerr,"extract_msg(): insufficient memory to store header.");
            msg_size = -2L; /* Fatal error */
            goto ExitFunct;
          }
          current = current->next;
          current->next = NULL;
          strcpy(current->info,msgbuf);
          current->used = NO;
        }
      }
    }

    /* Check for end of message, stop reading if end. */
    if (curpos == endpos) break;

    /* Check for article over-run. If we hit this, it's a bad thing.
       The reader gave us an article with binary headers. */

    if (curpos > endpos)
    {
      strcpy(_slrerr,"extract_msg(): message file out of sync on header.");
      msg_size = -2L; /* Fatal error */
      goto ExitFunct;
    }
  }

HeaderDone:

  /* Check message size. A zero here means the message
     received from the reader is bad. */

  if (msg_size == 0L)
  {
    sprintf(_slrerr,"extract_msg(): found zero byte message.");
    msg_size = -2L; /* Fatal error */
    goto ExitFunct;
  }

  /* Write the message header to temporary reply file */

  if ((work_file = fopen(REPLY_TMP,"wb")) == NULL)
  {
    sprintf(_slrerr,"extract_msg(): cannot open work file.");
    msg_size = -2L; /* Fatal error */
    goto ExitFunct;
  }
  if (dump_header(header,work_file) != 0)
  {
    msg_size = -2L; /* Fatal error */
    goto ExitFunct;
  }

  /* Now we need to write the message body to the temp file in
     binary format untouched. Note, as of Waffle 1.65, a real binary
     message will break Waffle, but we'll sure pass it along so as
     to be compliant with SOUP 1.2 */

  fprintf(work_file,"%s",msgbuf);
  curpos = ftell(msg_file);

  while (curpos != endpos)
  {
    if ((endpos - curpos) >= 1024)
    {
      fread(msgbuf,1024,1,msg_file);
    }
    else
    {
      fread(msgbuf,(endpos - curpos),1,msg_file);

      /* Whap a NULL to the end of the data stream. This is to
         insure we write actual message contents only, and not
         garbage stored in 'msgbuf'. */

      msgbuf[(endpos - curpos)] = '\0';
    }
    fwrite(msgbuf,strlen(msgbuf),1,work_file);
    curpos = ftell(msg_file);
  }

ExitFunct:
  if (work_file) fclose(work_file);
  if (header)
  {
    current = header;
    while (current != NULL)
    {
      header = current;
      current = current->next;
      free(header);
    }
    header = NULL;
  }
  if (scanlist) free(scanlist);
  {
    current = scanlist;
    while (current != NULL)
    {
      scanlist = current;
      current = current->next;
      free(scanlist);
    }
    scanlist = NULL;
  }
  if (current) free(current);
  return msg_size;
}

/*
 * Function: allow_header()
 * Purpose : Check header deny file for a match.
 * Return  : 0 to allow header, non-zero to deny header
*/

int allow_header(char *string)
{
  FILE *deny_file = NULL;
  char deny_path[MAXPATH];
  char deny_string[80];

  strcpy(deny_path,solar_path);
  strcat(deny_path,"\\");
  switch (msg_type) {
    /* Binary news message */
    case 'B'  : strcat(deny_path,NEWS_HEAD_DENY);
                break;
    /* Binary mail message */
    case 'b'  : strcat(deny_path,MAIL_HEAD_DENY);
                break;
  }
  if ((deny_file = fopen(deny_path,"rt")) == NULL)
  {
    goto AllowExit;
  }
  while (fgets(deny_string,80,deny_file) != NULL)
  {
    if (strnicmp(deny_string,string,strlen(deny_string) - 1) == 0)
    {
      fclose(deny_file);
      goto DenyExit;
    }
  }
  fclose(deny_file);
AllowExit:
  return 0;
DenyExit:
  return 1;
}

/*
 * Function: int dump_header(HEADER *header, FILE *work_file)
 * Purpose : Write header to work file.
 * Return  : zero on success, non-zero on error (sets _slrerr)
*/

int  dump_header(HEADER *header, FILE *work_file)
{
  /* Local Data */
  FILE   *order_file = NULL;
  HEADER *thisone = header;
  char   order_path[MAXPATH];
  char   orderbuf[80];
  char   date_time[35];

  strcpy(order_path,solar_path);
  strcat(order_path,"\\");
  switch (msg_type) {
    /* Binary news message */
    case 'B'  : strcat(order_path,NEWS_HEAD_ORDER);
                break;
    /* Binary mail message */
    case 'b'  : strcat(order_path,MAIL_HEAD_ORDER);
                break;
  }
  if (strcpy(date_time,rfctime()) == NULL)
  {
    strcpy(_slrerr,"extract_msg(): internal error creating rfctime.");
    goto ErrorExit;
  }
  if ((order_file = fopen(order_path,"rt")) == NULL)
  {
    goto ErrorExit;
  }
  while (fgets(orderbuf,80,order_file) != NULL)
  {
    while (thisone)
    {
      if (strnicmp(thisone->info,orderbuf,strlen(orderbuf) - 1) == 0) break;
      thisone = thisone->next;
    }
    if (thisone != NULL)
    {
      fprintf(work_file,"%s",thisone->info);
      thisone->used = YES;
    }
    else
    {
      if (strnicmp(orderbuf,"Path:",5) == 0)
        fprintf(work_file,"Path: %s!%s\n",uucp_name,username);

      if (strnicmp(orderbuf,"From:",5) == 0)
      {
        fprintf(work_file,"From: %s@%s",username,domain_name);
        if (strcmp(real_name,"NONE") != 0)
        {
          fprintf(work_file," (%s)",real_name);
        }
        fprintf(work_file,"\n");
      }
      if (strnicmp(orderbuf,"Date:",5) == 0)
        fprintf(work_file,"Date: %s\n",date_time);
      if (strnicmp(orderbuf,"Message-Id:",11) == 0)
        write_msgid(work_file);
      if ((strnicmp(orderbuf,"Organization:",13) == 0) && (strcmp(organization,"NONE") != 0))
        fprintf(work_file,"Organization: %s\n",organization);
    }
    thisone = header;
  }
  fclose(order_file);
  while (thisone)
  {
    if (thisone->used == NO)
    {
      fprintf(work_file,"%s",thisone->info);
    }
    thisone = thisone->next;
  }
GoodExit:
  return 0;
ErrorExit:
  return -1;
}

/*
 * Function: long binlen(char bigendian[4])
 * Purpose : Convert 4-byte field into 32-bit big endian value.
 * Return  : Length of message in bytes, or 0 on EOF or error.
*/

long binlen(char bigendian[4])
{
  return (((((long)(bigendian[0])) & 0xFF) << 24) |
      ((((long)(bigendian[1])) & 0xFF) << 16) |
      ((((long)(bigendian[2])) & 0xFF) <<  8) |
      ((((long)(bigendian[3])) & 0xFF)));
}

/*
 * Function: write_msgid()
 * Purpose : Write a unique Message-ID: header based on the time.
 * Return  : 0 on success, non-zero on error.
*/

int write_msgid(FILE *work_file)
{
  time_t t;
  struct tm *gmt;

  t = time(NULL);
  gmt = gmtime(&t);

  fprintf(work_file,"Message-Id: <%u%u%u.%u%u.s%ss@%s>\n",gmt->tm_hour,  \
          gmt->tm_min,gmt->tm_sec,gmt->tm_yday,gmt->tm_year,MSGID_VERSION, \
          domain_name);

  return 0;
}

/*
 * Function: spam_count()
 * Purpose : Count number of , separated items on header line.
 * Return  : Number of , separated items.
*/

long spam_count()
{
  long group_count  = 0L;
  char msgbuf_temp[BUFSIZE];
  char *p = NULL;

  strcpy(msgbuf_temp,msgbuf);

  p = strtok(msgbuf_temp,",");
  while (!p)
  {
    group_count++;
    p = strtok(NULL,",");
  }
  return group_count;
}
